--[[
    TrailKit 1.0
]]
tm.os.SetModTargetDeltaTime(1/60)

-- =================================================================
-- MODULE: LOGGER
-- =================================================================
_G.UnifiedLogConfig = {
    Debug = false,    -- Set to true to see [DEBUG] messages
    SpamLimit = 1.0,  -- Seconds to suppress identical messages
}

local UnifiedLogger = {
    _history = {}
}

function UnifiedLogger.log(level, message)
    if level == "DEBUG" and not _G.UnifiedLogConfig.Debug then return end
    
    local msg = tostring(message)
    local now = tm.os.GetTime()
    
    -- Check spam suppression
    local lastTime = UnifiedLogger._history[msg]
    if lastTime and (now - lastTime < _G.UnifiedLogConfig.SpamLimit) then
        return -- Suppressed
    end
    UnifiedLogger._history[msg] = now
    
    tm.os.Log("[" .. level .. "] " .. msg)
end

function UnifiedLogger.info(msg) UnifiedLogger.log("INFO", msg) end
function UnifiedLogger.warn(msg) UnifiedLogger.log("WARN", msg) end
function UnifiedLogger.error(msg) UnifiedLogger.log("ERROR", msg) end
function UnifiedLogger.debug(msg) UnifiedLogger.log("DEBUG", msg) end

_G.Logger = UnifiedLogger

-- =================================================================
-- MODULE: CORE / UPDATE MANAGER
-- =================================================================
local UpdateManager = {
    _callbacks = {}
}

function UpdateManager.register(callback, priority)
    if type(callback) == "function" then
        table.insert(UpdateManager._callbacks, {
            func = callback,
            priority = priority or 100
        })
    end
end

function UpdateManager.initialize()
    table.sort(UpdateManager._callbacks, function(a, b)
        return a.priority < b.priority
    end)
    
    local usersUpdate = _G.update
    local firstFrame = true
    
    _G.update = function()
        if firstFrame then
            firstFrame = false
            if type(_G.defineDialogues) == "function" then pcall(_G.defineDialogues) end
            if type(_G.RegisterChatCommands) == "function" then pcall(_G.RegisterChatCommands) end
            if type(_G.OnModStart) == "function" then pcall(_G.OnModStart) end
            
            Logger.debug("Auto-initialization complete.")
        end

        if usersUpdate then
            pcall(usersUpdate)
        end
        
        for _, cbInfo in ipairs(UpdateManager._callbacks) do
            local success, err = pcall(cbInfo.func)
            if not success then
                Logger.error("UpdateManager Exception: " .. tostring(err))
            end
        end
    end
end

_G.UpdateManager = UpdateManager

-- =================================================================
-- SHARED HELPERS
-- =================================================================
local function truncate(str, len)
    if type(str) ~= "string" then return "" end
    len = len or 32
    if #str > len then
        return str:sub(1, len - 3) .. "..."
    end
    return str
end

local function dist(p1, p2)
    if not p1 or not p2 then return math.huge end
    local dx, dy, dz = p1.x - p2.x, p1.y - p2.y, p1.z - p2.z
    return math.sqrt(dx*dx + dy*dy + dz*dz)
end

local function norm(v)
    return v:Normalized()
end

local clamp = function(val, min, max) return math.max(min, math.min(max, val)) end
local floor = math.floor
local sqrt = math.sqrt
local sin = math.sin
local cos = math.cos
local rad = math.rad
local deg = math.deg
local abs = math.abs
local acos = math.acos
local asin = math.asin

local function lerp(a, b, t)
    return a + (b - a) * clamp(t, 0, 1)
end

-- =================================================================
-- MODULE: MATH LIB
-- =================================================================
_G.MathLib = (function()
    local MathLib = {}

    function MathLib.clamp(val, min, max)
        if not val then return min end
        if val < min then return min end
        if val > max then return max end
        return val
    end

    function MathLib.lerp(a, b, t)
        return a + (b - a) * MathLib.clamp(t, 0, 1)
    end

    function MathLib.lerpUnclamped(a, b, t)
        return a + (b - a) * t
    end

    function MathLib.round(num, decimals)
        local mult = 10^(decimals or 0)
        return math.floor(num * mult + 0.5) / mult
    end

    function MathLib.map(value, inMin, inMax, outMin, outMax)
        return outMin + (value - inMin) * (outMax - outMin) / (inMax - inMin)
    end

    function MathLib.sign(num)
        return num > 0 and 1 or (num < 0 and -1 or 0)
    end

    function MathLib.randomRange(min, max)
        return min + math.random() * (max - min)
    end
    
    function MathLib.distance(x1, y1, x2, y2)
        return math.sqrt((x2 - x1)^2 + (y2 - y1)^2)
    end
    
    return MathLib
end)()

-- =================================================================
-- MODULE: EVENT LIB
-- =================================================================
_G.EventLib = (function()
    local EventLib = {}
    local listeners = {}

    function EventLib.on(eventName, handler)
        if not eventName or type(handler) ~= "function" then return end
        listeners[eventName] = listeners[eventName] or {}
        table.insert(listeners[eventName], handler)
    end

    function EventLib.off(eventName, handler)
        if not eventName or not listeners[eventName] then return end
        for i, h in ipairs(listeners[eventName]) do
            if h == handler then
                table.remove(listeners[eventName], i)
                return
            end
        end
    end

    function EventLib.emit(eventName, ...)
        if not eventName or not listeners[eventName] then return end
        for _, handler in ipairs(listeners[eventName]) do
            local success, err = pcall(handler, ...)
            if not success then
                tm.os.Log("EventLib [ERROR]: Handler for '"..eventName.."' failed: " .. tostring(err))
            end
        end
    end

    function EventLib.once(eventName, handler)
        local wrapper
        wrapper = function(...)
            EventLib.off(eventName, wrapper)
            handler(...)
        end
        EventLib.on(eventName, wrapper)
    end
    
    function EventLib.clear(eventName)
        if eventName then listeners[eventName] = nil else listeners = {} end
    end

    return EventLib
end)()

-- =================================================================
-- MODULE: TIMER LIB
-- =================================================================
_G.TimerLib = (function()
    local TimerLib = {}

    local UpdateManager = _G.UpdateManager

    local activeTimers = {}
    local timerGroups = {}
    local nextTimerId = 1
    local timersToRemove = {}

    -- =================================================================
    -- HELPER FUNCTIONS
    -- =================================================================

    local function _getCurrentTime(useGameTime)
        return useGameTime and tm.os.GetTime() or tm.os.GetRealtimeSinceStartup()
    end

    local function _createTimer(options)
        local id = options.id or nextTimerId
        nextTimerId = nextTimerId + 1

        local useGameTime = options.useGameTime or false
        local currentTime = _getCurrentTime(useGameTime)
        
        local timer = {
            id = id,
            isRepeating = options.isRepeating,
            isPaused = false,
            duration = options.interval or options.duration,
            callback = options.isRepeating and options.onTick or options.onComplete,
            startTime = currentTime,
            endTime = currentTime + (options.interval or options.duration),
            pauseTimeRemaining = 0,
            useGameTime = useGameTime,
            totalReps = options.count or (options.isRepeating and -1 or 1),
            currentRep = 0,
            group = options.group
        }

        activeTimers[id] = timer

        if options.group then
            if not timerGroups[options.group] then
                timerGroups[options.group] = {}
            end
            timerGroups[options.group][id] = true
        end

        return id
    end

    -- =================================================================
    -- PUBLIC API
    -- =================================================================

    function TimerLib.after(options)
        if type(options) ~= "table" or type(options.duration) ~= "number" or type(options.onComplete) ~= "function" then
            tm.os.Log("TimerLib [ERROR]: TimerLib.after requires an options table with 'duration' (number) and 'onComplete' (function).")
            return nil
        end

        options.isRepeating = false
        return _createTimer(options)
    end

    function TimerLib.every(options)
        if type(options) ~= "table" or type(options.interval) ~= "number" or type(options.onTick) ~= "function" then
            tm.os.Log("TimerLib [ERROR]: TimerLib.every requires an options table with 'interval' (number) and 'onTick' (function).")
            return nil
        end

        options.isRepeating = true
        return _createTimer(options)
    end

    function TimerLib.cancel(timerId)
        local timer = activeTimers[timerId]
        if timer then
            if timer.group and timerGroups[timer.group] then
                timerGroups[timer.group][timerId] = nil
            end
            activeTimers[timerId] = nil
            return true
        end
        return false
    end

    function TimerLib.pause(timerId)
        local timer = activeTimers[timerId]
        if timer and not timer.isPaused then
            timer.isPaused = true
            timer.pauseTimeRemaining = timer.endTime - _getCurrentTime(timer.useGameTime)
            return true
        end
        return false
    end

    function TimerLib.resume(timerId)
        local timer = activeTimers[timerId]
        if timer and timer.isPaused then
            timer.isPaused = false
            timer.endTime = _getCurrentTime(timer.useGameTime) + timer.pauseTimeRemaining
            timer.pauseTimeRemaining = 0
            return true
        end
        return false
    end

    function TimerLib.getTimeRemaining(timerId)
        local timer = activeTimers[timerId]
        if timer then
            return timer.isPaused and timer.pauseTimeRemaining or math.max(0, timer.endTime - _getCurrentTime(timer.useGameTime))
        end
        return nil
    end

    function TimerLib.isPaused(timerId)
        local timer = activeTimers[timerId]
        return timer and timer.isPaused
    end

    function TimerLib.cancelAll()
        activeTimers = {}
        timerGroups = {}
        tm.os.Log("TimerLib [INFO]: All active timers and groups have been cancelled.")
    end

    -- =================================================================
    -- GROUP CONTROL FUNCTIONS
    -- =================================================================

    function TimerLib.pauseGroup(groupName)
        if timerGroups[groupName] then
            for id, _ in pairs(timerGroups[groupName]) do
                TimerLib.pause(id)
            end
        end
    end

    function TimerLib.resumeGroup(groupName)
        if timerGroups[groupName] then
            for id, _ in pairs(timerGroups[groupName]) do
                TimerLib.resume(id)
            end
        end
    end

    function TimerLib.cancelGroup(groupName)
        if timerGroups[groupName] then
            local idsToCancel = {}
            for id, _ in pairs(timerGroups[groupName]) do
                table.insert(idsToCancel, id)
            end
            for _, id in ipairs(idsToCancel) do
                TimerLib.cancel(id)
            end
            timerGroups[groupName] = nil
        end
    end

    -- =================================================================
    -- CORE UPDATE LOOP
    -- =================================================================

    function TimerLib.update()
        if next(activeTimers) == nil then return end

        local realTime = tm.os.GetRealtimeSinceStartup()
        local gameTime = tm.os.GetTime()

        for id, timer in pairs(activeTimers) do
            if not timer.isPaused then
                local currentTime = timer.useGameTime and gameTime or realTime
                if currentTime >= timer.endTime then
                    timer.currentRep = timer.currentRep + 1
                    
                    pcall(timer.callback, timer.id, timer.currentRep)

                    if timer.isRepeating then
                        if timer.totalReps > 0 and timer.currentRep >= timer.totalReps then
                            table.insert(timersToRemove, id)
                        else
                            -- To avoid drift, we add the duration to the previous endTime, not the current time.
                            timer.endTime = timer.endTime + timer.duration
                        end
                    else
                        table.insert(timersToRemove, id)
                    end
                end
            end
        end

        if #timersToRemove > 0 then
            for _, id in ipairs(timersToRemove) do
                TimerLib.cancel(id)
            end
            timersToRemove = {}
        end
    end

    -- =================================================================
    -- FINAL REGISTRATION
    -- =================================================================

    UpdateManager.register(TimerLib.update, 100)

    return TimerLib
end)()

-- =================================================================
-- MODULE: SAVE LIB
-- =================================================================
_G.SaveLib = (function()
    local SaveLib = {}
    
    local json = _G.json
    local TimerLib = _G.TimerLib
    
    local saveCache = {}
    local pendingSaves = {}

    local function _parsePath(path)
        local keys = {}
        for key in string.gmatch(path, "[^%.]+") do
            table.insert(keys, key)
        end
        return keys
    end

    function SaveLib.save(options)
        if not options or not options.id or not options.data then return false end

        local success, result = pcall(json.serialize, options.data)
        if not success then
            tm.os.Log("SaveLib [ERROR]: Failed to serialize data for id: " .. tostring(options.id) .. " | Reason: " .. tostring(result))
            return false
        end
        local jsonString = result

        local mainPath = options.id .. ".json"
        local backupPath = options.id .. ".json.bak"

        local oldContent = tm.os.ReadAllText_Dynamic(mainPath)
        if oldContent and oldContent ~= "" then
            pcall(tm.os.WriteAllText_Dynamic, backupPath, oldContent)
        end

        pcall(tm.os.WriteAllText_Dynamic, mainPath, jsonString)
        
        saveCache[options.id] = options.data
        
        return true
    end

    function SaveLib.queueSave(options)
        if not options or not options.id or not options.data then return false end

        local delay = options.delay or 2.0

        pendingSaves[options.id] = options.data

        local timerId = "save_queue_" .. options.id
        TimerLib.cancel(timerId)

        TimerLib.after({
            id = timerId,
            duration = delay,
            onComplete = function()
                if pendingSaves[options.id] then
                    SaveLib.save({ id = options.id, data = pendingSaves[options.id] })
                    pendingSaves[options.id] = nil
                    tm.os.Log("SaveLib [INFO]: Queued save executed for id '" .. options.id .. "'.")
                end
            end,
            group = "SAVE_QUEUE_JOBS"
        })
        return true
    end


    function SaveLib.load(options)
        if not options or not options.id then return options.default or nil end

        if saveCache[options.id] then
            return saveCache[options.id]
        end

        local mainPath = options.id .. ".json"
        local backupPath = options.id .. ".json.bak"
        
        local jsonString = tm.os.ReadAllText_Dynamic(mainPath)

        if jsonString == nil or jsonString == "" then
            return options.default or nil
        end

        local success, dataTable = pcall(json.parse, jsonString)

        if success then
            saveCache[options.id] = dataTable
            return dataTable
        else
            tm.os.Log("SaveLib [WARNING]: Corrupt save file for '" .. tostring(options.id) .. "'. Trying backup.")
            local backupJsonString = tm.os.ReadAllText_Dynamic(backupPath)
            if backupJsonString and backupJsonString ~= "" then
                local backupSuccess, backupDataTable = pcall(json.parse, backupJsonString)
                if backupSuccess then
                    tm.os.Log("SaveLib [INFO]: Restored from backup for '" .. tostring(options.id) .. "'.")
                    pcall(tm.os.WriteAllText_Dynamic, mainPath, backupJsonString)
                    saveCache[options.id] = backupDataTable
                    return backupDataTable
                else
                    tm.os.Log("SaveLib [ERROR]: Backup file for '" .. tostring(options.id) .. "' is also corrupt.")
                    return options.default or nil
                end
            else
                tm.os.Log("SaveLib [WARNING]: No backup file found for '" .. tostring(options.id) .. "'.")
                return options.default or nil
            end
        end
    end

    function SaveLib.delete(options)
        if not options or not options.id then return false end
        local mainPath = options.id .. ".json"
        local backupPath = options.id .. ".json.bak"
        pcall(tm.os.WriteAllText_Dynamic, mainPath, "{}")
        pcall(tm.os.WriteAllText_Dynamic, backupPath, "")
        
        saveCache[options.id] = nil
        
        return true
    end

    function SaveLib.clearCache(options)
        options = options or {}
        if options.id then
            saveCache[options.id] = nil
            tm.os.Log("SaveLib [INFO]: Cleared cache for id '" .. options.id .. "'.")
        else
            saveCache = {}
            tm.os.Log("SaveLib [INFO]: Cleared entire save cache.")
        end
    end

    function SaveLib.exists(options)
        if not options or not options.id then return false end
        if saveCache[options.id] then return true end
        
        local content = tm.os.ReadAllText_Dynamic(options.id .. ".json")
        return content and content ~= "" and content ~= "{}"
    end

    function SaveLib.getValue(options)
        if not options or not options.id or not options.key then return options.default or nil end
        local data = SaveLib.load({ id = options.id, default = {} })
        local keys = _parsePath(options.key)
        local current = data
        for i = 1, #keys do
            local key = keys[i]
            if type(current) ~= "table" or current[key] == nil then
                return options.default or nil
            end
            current = current[key]
        end
        return current
    end

    function SaveLib.updateValue(options)
        if not options or not options.id or not options.key then return false end
        local data = SaveLib.load({ id = options.id, default = {} })
        local keys = _parsePath(options.key)
        local current = data
        for i = 1, #keys - 1 do
            local key = keys[i]
            if type(current[key]) ~= "table" then
                current[key] = {}
            end
            current = current[key]
        end
        current[keys[#keys]] = options.value
        return SaveLib.save({ id = options.id, data = data })
    end

    function SaveLib.createAutosave(options)
        if not options or not options.id or not options.interval or not options.getDataCallback then
            tm.os.Log("SaveLib [ERROR]: createAutosave requires id, interval, and getDataCallback.")
            return nil
        end

        local onTickCallback = function()
            local dataToSave = options.getDataCallback()
            if dataToSave then
                SaveLib.save({ id = options.id, data = dataToSave })
                tm.os.Log("SaveLib [INFO]: Autosaved data for id '" .. options.id .. "'.")
            end
        end

        return TimerLib.every({
            id = "autosave_" .. options.id,
            interval = options.interval,
            onTick = onTickCallback,
            group = "AUTOSAVE_JOBS"
        })
    end

    function SaveLib.removeAutosave(id)
        return TimerLib.cancel("autosave_" .. id)
    end

    return SaveLib
end)()

-- =================================================================
-- MODULE: PLAYER LIB
-- =================================================================
_G.PlayerLib = (function()
    local PlayerLib = {}
    
    -- Local References
    local TimerLib = _G.TimerLib
    local SaveLib = _G.SaveLib
    local UpdateManager = _G.UpdateManager

    -- =================================================================
    -- INTERNAL STATE
    -- =================================================================

    local players = {}
    local spawnLocations = {}
    local PlayerMT = { __index = {} }
    local _respawnRequests = {} 

    -- =================================================================
    -- CONFIGURATION
    -- =================================================================
    PlayerLib.config = {
        teleportDistanceThresholdSq = 22500, -- (150 * 150)
        teleportTimeThreshold = 0.1,
        teleportSpeedThresholdSq = 40000,    -- (200 * 200)

        respawnCooldown = 2.0,
        respawnHoldDuration = 1.4,
        respawnKeyName = "backspace"
    }

    -- =================================================================
    -- EVENT HANDLING
    -- =================================================================
    PlayerLib.events = {
        onPlayerJoined = {},
        onPlayerLeft = {},
        onEnterSeat = {},
        onLeaveSeat = {},
        onEnterBuildMode = {},
        onLeaveBuildMode = {},
        onTeleport = {},
        onRespawn = {},
        onTeamChange = {},
        onStateChange = {} -- Event for state changes
    }

    local function fireEvent(eventName, ...)
        if _G.EventLib then _G.EventLib.emit(eventName, ...) end
        if PlayerLib.events[eventName] then
            for _, callback in ipairs(PlayerLib.events[eventName]) do
                pcall(callback, ...)
            end
        end
    end

    -- =================================================================
    -- PLAYER OBJECT METHODS
    -- =================================================================

    function PlayerMT.__index:isAlive() return self._isAlive end
    function PlayerMT.__index:getName() return self.name end
    function PlayerMT.__index:getGameObject() return self.gameObject end
    function PlayerMT.__index:getTransform() return self.transform end
    function PlayerMT.__index:getPosition() return self.transform:GetPositionWorld() end
    function PlayerMT.__index:getRotation() return self.transform:GetRotationWorld() end
    function PlayerMT.__index:getTeam() return self.lastTeamId end
    function PlayerMT.__index:isSeated() return self.isInSeat end
    function PlayerMT.__index:isInBuildMode() return self.isInBuildMode end
    function PlayerMT.__index:getStructure() return self.structure end
    function PlayerMT.__index:getSeatBlock() return self.seatBlock end
    function PlayerMT.__index:getVelocity()
        if self.structure and self.structure:Exists() then
            return self.structure:GetVelocity()
        end
        return tm.vector3.Create(0,0,0)
    end
    function PlayerMT.__index:getSpeed()
        if self.structure and self.structure:Exists() then
            return self.structure:GetSpeed()
        end
        return 0
    end
    
    function PlayerMT.__index:getBuildStructures()
        return tm.players.GetPlayerStructuresInBuild(self.id)
    end
    
    function PlayerMT.__index:getSelectedBlock()
        local block = tm.players.GetPlayerSelectBlockInBuild(self.id)
        if block and _G.PhysicsLib and _G.PhysicsLib.Block then
            return _G.PhysicsLib.Block(block)
        end
        return block
    end

    -- Data Management
    function PlayerMT.__index:hasDataValue(key) return self.data[key] ~= nil end
    function PlayerMT.__index:setDataValue(key, value) self.data[key] = value end
    function PlayerMT.__index:getDataValue(key, defaultValue) return self.data[key] or defaultValue end
    function PlayerMT.__index:incrementDataValue(key, amount)
        local current = self:getDataValue(key, 0)
        if type(current) == "number" then self:setDataValue(key, current + (amount or 1)) end
    end
    function PlayerMT.__index:setPersistence(shouldPersist)
        self.isPersistent = (shouldPersist == true)
        if self.isPersistent then self:loadPersistentData() end
    end
    function PlayerMT.__index:savePersistentData()
        if not self.isPersistent then return false end
        return SaveLib.save({ id = self:getSlotName(), data = self.data })
    end
    function PlayerMT.__index:loadPersistentData()
        if not self.isPersistent then return end
        self.data = SaveLib.load({ id = self:getSlotName(), default = {} })
    end
    function PlayerMT.__index:clearPersistentData()
        if not self.isPersistent then return false end
        self.data = {}
        return SaveLib.delete({ id = self:getSlotName() })
    end
    function PlayerMT.__index:getSlotName()
        if self.isPersistent then
            return "playerdata/persistent/player_" .. self.profileId
        else
            return "playerdata/temporary/player_" .. self.id
        end
    end

    -- Player Actions
    function PlayerMT.__index:kill() tm.players.KillPlayer(self.id) end
    function PlayerMT.__index:setInvincible(isInvincible) tm.players.SetPlayerIsInvincible(self.id, isInvincible) end
    function PlayerMT.__index:setJetpack(isEnabled) tm.players.SetJetpackEnabled(self.id, isEnabled) end
    function PlayerMT.__index:setTeam(teamId) tm.players.SetPlayerTeam(self.id, teamId) end
    function PlayerMT.__index:setBuilderEnabled(isEnabled) tm.players.SetBuilderEnabled(self.id, isEnabled) end
    function PlayerMT.__index:setRepairEnabled(isEnabled) tm.players.SetRepairEnabled(self.id, isEnabled) end
    function PlayerMT.__index:spawnStructure(blueprintName, structureId, position, rotation) tm.players.SpawnStructure(self.id, blueprintName, structureId, position, rotation) end
    function PlayerMT.__index:enterSeat(structureId) tm.players.PlacePlayerInSeat(self.id, structureId) end
    function PlayerMT.__index:despawnAllStructures()
        local playerStructures = tm.players.GetPlayerStructures(self.id)
        for _, s in ipairs(playerStructures) do
            if s and s:Exists() then
                s:Dispose() 
            end
        end
    end
    function PlayerMT.__index:teleport(position, rotation)
        local spawnId = "temp_teleport_" .. self.id
        tm.players.SetSpawnPoint(self.id, spawnId, position, rotation or tm.vector3.Create(0,0,0))
        tm.players.TeleportPlayerToSpawnPoint(self.id, spawnId, false)
    end
    function PlayerMT.__index:setRespawnLocation(locationName)
        if spawnLocations[locationName] then
            tm.players.SetPlayerSpawnLocation(self.id, locationName)
            return true
        else
            tm.os.Log("PlayerLib [WARNING]: Attempted to set respawn to non-existent location '"..locationName.."' for player "..self.name)
            return false
        end
    end

    function PlayerMT.__index:setSpawnPoint(locationId, pos, rot)
        pcall(tm.players.SetSpawnPoint, self.id, locationId, pos, rot or tm.vector3.Create(0,0,0))
    end

    function PlayerMT.__index:teleportToSpawnPoint(locationId, keepVelocity)
        pcall(tm.players.TeleportPlayerToSpawnPoint, self.id, locationId, keepStructures or false)
    end

    -- State Management
    function PlayerMT.__index:setState(newState, duration)
        local oldState = self.state
        if oldState == newState then return end

        -- Cancel any existing timer that would have reverted a temporary state
        if self.stateTimerId then
            TimerLib.cancel(self.stateTimerId)
            self.stateTimerId = nil
        end

        self.state = newState or "default"
        fireEvent("onStateChange", self, self.state, oldState)

        -- If a duration is provided, set a timer to revert the state to "default"
        if duration and duration > 0 then
            self.stateTimerId = TimerLib.after({
                duration = duration,
                onComplete = function()
                    -- Check if the state is still the one we set before reverting
                    if self.state == newState then
                        self:setState("default")
                    end
                end
            })
        end
    end
    function PlayerMT.__index:getState() return self.state or "default" end
    function PlayerMT.__index:isInState(stateName) return self.state == stateName end

    -- =================================================================
    -- DX: WRAPPER METHODS (Player API)
    -- =================================================================

    function PlayerMT.__index:hasPermission(level)
        if _G.PermissionsLib then
            return _G.PermissionsLib.hasPermission(self, level)
        end
        return false
    end

    function PlayerMT.__index:grantPermission(level)
        if _G.PermissionsLib then
            _G.PermissionsLib.grantPermission(self, level)
        end
    end

    function PlayerMT.__index:toast(message, duration)
        if _G.UILib then
            _G.UILib.createSubtleMessage({ 
                id = "toast_" .. self.id .. "_" .. tostring(tm.os.GetTime()), 
                header = "Info", 
                message = message, 
                target = self.id,
                options = { duration = duration or 3 }
            })
        end
    end

    function PlayerMT.__index:message(text, color)
        if _G.InputLib then
            _G.InputLib.messagePlayer({ player = self, message = text, color = color })
        end
    end

    function PlayerMT.__index:alert(header, message, duration)
        pcall(tm.playerUI.ShowIntrusiveMessageForPlayer, self.id, header, message, duration or 3)
    end
    
    function PlayerMT.__index:getSeat()
        return tm.players.GetPlayerSeatBlock(self.id)
    end
    
    function PlayerMT.__index:getPrimaryColor()
        return tm.players.GetPrimaryBlockColor(self.id)
    end
    
    function PlayerMT.__index:getSecondaryColor()
        return tm.players.GetSecondaryBlockColor(self.id)
    end

    function PlayerMT.__index:startDialogue(dialogueId, sourceObject)
        if _G.DialogueLib then
            _G.DialogueLib.start({ target = self.id, dialogueId = dialogueId, lookAt = sourceObject })
        end
    end

    function PlayerMT.__index:getStructures()
        return tm.players.GetPlayerStructures(self.id)
    end

    function PlayerMT.__index:getBuildStructures()
        return tm.players.GetPlayerStructuresInBuild(self.id)
    end

    function PlayerMT.__index:getSelectedBlock()
        return tm.players.GetPlayerSelectBlockInBuild(self.id)
    end
    
    function PlayerMT.__index:isBuilderEnabled()
        return tm.players.GetBuilderEnabled(self.id)
    end

    function PlayerMT.__index:isRepairEnabled()
        return tm.players.GetRepairEnabled(self.id)
    end

    function PlayerMT.__index:setCameraPos(pos)
        pcall(tm.players.SetCameraPosition, self.id, pos)
    end

    function PlayerMT.__index:setCameraRot(rot)
        pcall(tm.players.SetCameraRotation, self.id, rot)
    end

    function PlayerMT.__index:activateCamera(fade)
        pcall(tm.players.ActivateCamera, self.id, fade or 0)
    end

    function PlayerMT.__index:deactivateCamera(fade)
        pcall(tm.players.DeactivateCamera, self.id, fade or 0)
    end

    function PlayerMT.__index:spawnStructure(blueprint, structureId, pos, rot)
        pcall(tm.players.SpawnStructure, self.id, blueprint, structureId, pos, rot)
    end

    function PlayerMT.__index:sitIn(structureId)
        pcall(tm.players.PlacePlayerInSeat, self.id, structureId)
    end
    
    function PlayerMT.__index:setData(key, value)
        if _G.SaveLib then
            self.data[key] = value
            if self.isPersistent then
                _G.SaveLib.updateValue({ id = self.profileId, key = key, value = value })
            end
        else
             self.data[key] = value
        end
    end
    
    function PlayerMT.__index:getData(key)
        if _G.SaveLib and self.isPersistent then
             local val = _G.SaveLib.getValue({ id = self.profileId, key = key })
             if val ~= nil then return val end
        end
        return self.data[key]
    end

    function PlayerMT.__index:makePersistent()
        self.isPersistent = true
        if _G.SaveLib then
            local loaded = _G.SaveLib.load({ id = self.profileId })
            if loaded then 
                for k,v in pairs(loaded) do self.data[k] = v end
            end
        end
    end

    function PlayerMT.__index:save()
        if _G.SaveLib and self.isPersistent then
             _G.SaveLib.save({ id = self.profileId, data = self.data })
        end
    end

    function PlayerMT.__index:setTeam(teamId)
        pcall(tm.players.SetPlayerTeam, self.id, teamId)
    end

    function PlayerMT.__index:setTeam(teamId)
        pcall(tm.players.SetPlayerTeam, self.id, teamId)
    end

    function PlayerMT.__index:kick()
        pcall(tm.players.Kick, self.id)
    end

    function PlayerMT.__index:kill()
        pcall(tm.players.KillPlayer, self.id)
    end

    function PlayerMT.__index:setInvincible(enabled)
        pcall(tm.players.SetPlayerIsInvincible, self.id, enabled)
    end

    function PlayerMT.__index:setJetpack(enabled)
        pcall(tm.players.SetJetpackEnabled, self.id, enabled)
    end

    function PlayerMT.__index:setBuilder(enabled)
        pcall(tm.players.SetBuilderEnabled, self.id, enabled)
    end

    function PlayerMT.__index:setRepair(enabled)
        pcall(tm.players.SetRepairEnabled, self.id, enabled)
    end

    function PlayerMT.__index:teleport(locationName)
        if _G.PlayerLib then
            _G.PlayerLib.teleportToLocation(locationName, self.id)
        end
    end

    function PlayerMT.__index:onKey(key, callback, onUp)
        if _G.InputLib then
            local action = onUp and _G.InputLib.onKeyUp or _G.InputLib.onKeyDown
            action(key, function(playerId)
                if playerId == self.id then callback() end
            end)
        end
    end
    
    function PlayerMT.__index:raycast(maxDistance, options)
        options = options or {}
        options.origin = self.transform:GetPositionWorld()
        options.direction = self.transform:GetRotationWorld():Multiply(tm.vector3.Create(0,0,1))
        options.maxDistance = maxDistance
        if _G.SceneLib then
            return _G.SceneLib.raycast(options)
        end
        return { hit = false }
    end

    function PlayerMT.__index:after(duration, callback)
        if _G.TimerLib then
            _G.TimerLib.after({
                duration = duration,
                onComplete = function()
                    if self:isAlive() then callback() end
                end
            })
        end
    end

    function PlayerMT.__index:playSound(soundName)
        if _G.SoundLib then
            _G.SoundLib.playAtPosition({ sound = soundName, position = self:getPosition() })
        end
    end

    -- UI Wrappers
    function PlayerMT.__index:createLabel(id, text)
        if _G.UILib then
            _G.UILib.createLabel({ target = self.id, id = id, text = text })
        end
    end

    function PlayerMT.__index:createButton(id, text, callback, data)
        if _G.UILib then
            _G.UILib.createButton({ target = self.id, id = id, text = text, callback = callback, data = data })
        end
    end

    function PlayerMT.__index:createInputField(id, defaultText, callback, data)
        if _G.UILib then
            _G.UILib.createInputField({ target = self.id, id = id, defaultText = defaultText, callback = callback, data = data })
        end
    end

    function PlayerMT.__index:updateUI(id, text)
        if _G.UILib then
            _G.UILib.updateText({ target = self.id, id = id, text = text })
        end
    end

    function PlayerMT.__index:removeUI(id)
        if _G.UILib then
            _G.UILib.removeElement({ target = self.id, id = id })
        end
    end

    function PlayerMT.__index:clearUI()
        if _G.UILib then
            _G.UILib.clearAll({ target = self.id })
        end
    end

    -- Camera Wrappers
    function PlayerMT.__index:lookAt(target, duration, options)
        if _G.CameraLib then
            options = options or {}
            options.target = self.id
            options.lookAtTarget = target
            options.duration = duration
            _G.CameraLib.lookAt(options)
        end
    end
    
    function PlayerMT.__index:follow(target, options)
        if _G.CameraLib then
            options = options or {}
            options.target = self.id
            options.followTarget = target
            _G.CameraLib.follow(options)
        end
    end

    function PlayerMT.__index:resetCamera()
        if _G.CameraLib then
            _G.CameraLib.release({ target = self.id })
        end
    end
    
    function PlayerMT.__index:playCinematic(filenames, options)
        if _G.CameraLib then
            options = options or {}
            options.target = self.id
            options.filenames = filenames
            _G.CameraLib.playCinematic(options)
        end
    end

    function PlayerMT.__index:spawnVehicle(options)
        if _G.PhysicsLib then
            options = options or {}
            options.playerId = self.id
            _G.PhysicsLib.spawnStructureWithPath(options)
        end
    end

    -- =================================================================
    -- PUBLIC API - PLAYER MANAGEMENT
    -- =================================================================

    local function getOrCreatePlayer(playerId)
        if players[playerId] then
            return players[playerId]
        end

        local playerTransform = tm.players.GetPlayerTransform(playerId)
        local playerObject = {
            id = playerId,
            name = tm.players.GetPlayerName(playerId) or "Unknown",
            profileId = tm.players.GetPlayerProfileId(playerId) or tostring(playerId),
            gameObject = tm.players.GetPlayerGameObject(playerId),
            transform = playerTransform,
            data = {},
            isPersistent = false,
            isInSeat = tm.players.IsPlayerInSeat(playerId),
            isInBuildMode = tm.players.GetPlayerIsInBuildMode(playerId),
            structure = nil,
            seatBlock = nil,
            lastPosition = playerTransform and playerTransform:GetPositionWorld() or tm.vector3.Create(0, 0, 0),
            _isAlive = true,
            isNew = true,
            lastCheckTime = tm.os.GetTime(),
            lastRespawnTime = 0,
            respawnFlag = 0,
            lastTeamId = tm.players.GetPlayerTeam(playerId),
            state = "default", -- Initialize player state
            stateTimerId = nil -- Timer for temporary states
        }
        setmetatable(playerObject, PlayerMT)
        players[playerId] = playerObject

        return playerObject
    end

    function PlayerLib.getPlayer(playerId) return getOrCreatePlayer(playerId) end

    function PlayerLib.getPlayerByName(name)
        if not name then return nil end
        local lowerName = name:lower()
        for _, p in pairs(PlayerLib.getAllPlayers()) do
            if p.name:lower() == lowerName then
                return p
            end
        end
        return nil
    end

    function PlayerLib.getPlayerByProfileId(profileId)
        if not profileId then return nil end
        for _, p in pairs(PlayerLib.getAllPlayers()) do
            if p.profileId == profileId then
                return p
            end
        end
        return nil
    end

    function PlayerLib.getTargetPlayers(target)
        local players = {}
        if not target then return players end

        local targetType = type(target)
        if targetType == "string" and target:lower() == "all" then
            for _, p in ipairs(tm.players.CurrentPlayers()) do table.insert(players, p.playerId) end
        elseif targetType == "table" then
            for _, p in ipairs(target) do table.insert(players, (type(p) == 'number') and p or p.id) end
        elseif targetType == "number" then
            table.insert(players, target)
        elseif targetType == "userdata" or (targetType == "table" and target.id) then -- Handle player object
            table.insert(players, target.id)
        end
        return players
    end

    function PlayerLib.getAllPlayers()
        for _, playerInfo in ipairs(tm.players.CurrentPlayers()) do
            getOrCreatePlayer(playerInfo.playerId)
        end
        return players
    end

    function PlayerLib.getPlayersInTeam(teamId)
        local teamPlayers = {}
        for _, p in pairs(PlayerLib.getAllPlayers()) do
            if p.lastTeamId == teamId then
                table.insert(teamPlayers, p)
            end
        end
        return teamPlayers
    end

    -- =================================================================
    -- PUBLIC API - SPAWNPOINT MANAGEMENT
    -- =================================================================

    function PlayerLib.createSpawnLocation(locationName, spawnPoints)
        if not locationName or not spawnPoints or type(spawnPoints) ~= "table" then
            tm.os.Log("PlayerLib [ERROR]: createSpawnLocation requires a name and a table of spawn points.")
            return false
        end

        for i, sp in ipairs(spawnPoints) do
            local pos = sp.position or sp[1] or tm.vector3.Create(0,0,0)
            local rot = sp.rotation or sp[2] or tm.vector3.Create(0,0,1)
            local playerIndex = i - 1
            pcall(tm.players.SetSpawnPoint, playerIndex, locationName, pos, rot)
        end

        spawnLocations[locationName] = spawnPoints
        tm.os.Log("PlayerLib [INFO]: Created spawn location '"..locationName.."' with "..#spawnPoints.." points.")
        return true
    end

    function PlayerLib.teleportToLocation(locationName, targetPlayers, keepStructures)
        if not spawnLocations[locationName] then
            tm.os.Log("PlayerLib [ERROR]: Cannot teleport to non-existent location '"..locationName.."'")
            return
        end
        
        local playersToTeleport = PlayerLib.getTargetPlayers(targetPlayers)

        for _, pid in ipairs(playersToTeleport) do
            pcall(tm.players.TeleportPlayerToSpawnPoint, pid, locationName, keepStructures or false)
        end
    end

    function PlayerLib.removeSpawnLocation(locationName)
        spawnLocations[locationName] = nil
    end

    -- =================================================================
    -- PUBLIC API - EVENT REGISTRATION
    -- =================================================================

    function PlayerLib.onPlayerJoined(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onPlayerJoined, callback) end end
    function PlayerLib.onPlayerLeft(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onPlayerLeft, callback) end end
    function PlayerLib.onEnterSeat(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onEnterSeat, callback) end end
    function PlayerLib.onLeaveSeat(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onLeaveSeat, callback) end end
    function PlayerLib.onEnterBuildMode(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onEnterBuildMode, callback) end end
    function PlayerLib.onLeaveBuildMode(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onLeaveBuildMode, callback) end end
    function PlayerLib.onTeleport(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onTeleport, callback) end end
    function PlayerLib.onRespawn(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onRespawn, callback) end end
    function PlayerLib.onTeamChange(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onTeamChange, callback) end end
    function PlayerLib.onStateChange(callback) if type(callback) == "function" then table.insert(PlayerLib.events.onStateChange, callback) end end

    -- =================================================================
    -- INTERNAL: INPUT KEY HANDLING
    -- =================================================================

    function PlayerLib._onRespawnKeyDown(playerId)
        if _respawnRequests[playerId] then return end
        local playerObject = players[playerId]
        if not playerObject or not (tm.os.GetTime() - playerObject.lastRespawnTime > PlayerLib.config.respawnCooldown) then return end

        _respawnRequests[playerId] = TimerLib.after({
            duration = PlayerLib.config.respawnHoldDuration,
            onComplete = function()
                local pObj = players[playerId]
                if pObj then
                    fireEvent("onRespawn", pObj, "manual")
                    pObj.lastRespawnTime = tm.os.GetTime()
                    pObj.respawnFlag = 2
                end
                _respawnRequests[playerId] = nil
            end,
            useGameTime = true 
        })
    end

    function PlayerLib._onRespawnKeyUp(playerId)
        if _respawnRequests[playerId] then
            TimerLib.cancel(_respawnRequests[playerId])
            _respawnRequests[playerId] = nil
        end
    end

    -- =================================================================
    -- CORE UPDATE LOOP FOR EVENT DETECTION
    -- =================================================================

    function PlayerLib.update()
        local currentTime = tm.os.GetTime()

        for pid, p in pairs(players) do
            if not p.gameObject or not p.gameObject:Exists() then
                if p._isAlive then p._isAlive = false end
                p.gameObject = tm.players.GetPlayerGameObject(pid)
            end
            
            if p.gameObject then
                if p.respawnFlag == 1 then p.lastPosition = p.transform:GetPositionWorld() end

                if not p._isAlive then
                    p._isAlive = true
                    if not p.isNew and currentTime - p.lastRespawnTime > PlayerLib.config.respawnCooldown then
                        fireEvent("onRespawn", p, "natural")
                        p.lastRespawnTime = currentTime
                        p.respawnFlag = 2
                    end
                end

                local isNowInSeat = tm.players.IsPlayerInSeat(pid)
                local isNowInBuildMode = tm.players.GetPlayerIsInBuildMode(pid)
                local currentPosition = p.transform:GetPositionWorld()
                local timeSinceLastCheck = currentTime - p.lastCheckTime
                
                if timeSinceLastCheck > 0 and p.respawnFlag == 0 then
                    local distSq = tm.vector3.op_Subtraction(currentPosition, p.lastPosition):Magnitude()^2
                    if distSq > PlayerLib.config.teleportDistanceThresholdSq and timeSinceLastCheck < PlayerLib.config.teleportTimeThreshold then
                        if not p.isNew then fireEvent("onTeleport", p, p.lastPosition, currentPosition) end
                    end
                end
                
                local currentTeamId = tm.players.GetPlayerTeam(pid)
                if currentTeamId ~= p.lastTeamId then
                    fireEvent("onTeamChange", p, p.lastTeamId, currentTeamId)
                    p.lastTeamId = currentTeamId
                end

                if p.respawnFlag == 0 then
                    if isNowInSeat ~= p.isInSeat and not p.isNew then
                        fireEvent(isNowInSeat and "onEnterSeat" or "onLeaveSeat", p)
                    end
                    
                    if isNowInBuildMode ~= p.isInBuildMode and not p.isNew then
                        fireEvent(isNowInBuildMode and "onEnterBuildMode" or "onLeaveBuildMode", p)
                    end
                end

                p.isInSeat = isNowInSeat
                p.isInBuildMode = isNowInBuildMode
                p.lastPosition = currentPosition
                p.lastCheckTime = currentTime
                p.isNew = false
                if p.respawnFlag > 0 then p.respawnFlag = p.respawnFlag - 1 end

                if isNowInSeat then
                    p.seatBlock = tm.players.GetPlayerSeatBlock(pid)
                    if p.seatBlock and p.seatBlock:Exists() then p.structure = p.seatBlock:GetStructure() end
                else
                    p.structure = nil
                    p.seatBlock = nil
                end
            end
        end
    end

    -- =================================================================
    -- INTERNAL EVENT HANDLERS & INITIALIZATION
    -- =================================================================

    local function internal_OnPlayerJoined(playerInfo)
        local p = getOrCreatePlayer(playerInfo.playerId)
        pcall(tm.input.RegisterFunctionToKeyDownCallback, p.id, "PlayerLib_RespawnKeyDown", PlayerLib.config.respawnKeyName)
        pcall(tm.input.RegisterFunctionToKeyUpCallback, p.id, "PlayerLib_RespawnKeyUp", PlayerLib.config.respawnKeyName)
        fireEvent("onPlayerJoined", p)
    end

    local function internal_OnPlayerLeft(playerInfo)
        local playerObject = players[playerInfo.playerId]
        if playerObject then
            fireEvent("onPlayerLeft", playerObject)
            if playerObject.isPersistent then playerObject:savePersistentData() end
            if _respawnRequests[playerObject.id] then TimerLib.cancel(_respawnRequests[playerObject.id]) end
            _respawnRequests[playerObject.id] = nil
            players[playerObject.id] = nil
        end
    end

    local function initialize()
        tm.players.OnPlayerJoined.add(internal_OnPlayerJoined)
        tm.players.OnPlayerLeft.add(internal_OnPlayerLeft)

        if tm.players.CurrentPlayers then
            for _, player in ipairs(tm.players.CurrentPlayers()) do
                internal_OnPlayerJoined(player)
            end
        end
    end

    -- Explicit global registration for input callbacks
    _G.PlayerLib_RespawnKeyDown = function(playerId) if PlayerLib and PlayerLib._onRespawnKeyDown then PlayerLib._onRespawnKeyDown(playerId) end end
    _G.PlayerLib_RespawnKeyUp = function(playerId) if PlayerLib and PlayerLib._onRespawnKeyUp then PlayerLib._onRespawnKeyUp(playerId) end end

    initialize()
    UpdateManager.register(PlayerLib.update, 800)

    return PlayerLib
end)()

-- =================================================================
-- MODULE: GAME SINGLETON (Global Settings)
-- =================================================================
_G.Game = (function()
    local Game = {}
    
    function Game.setTimeScale(scale)
        pcall(tm.physics.SetTimeScale, scale)
    end
    
    function Game.getTimeScale()
        return tm.physics.GetTimeScale()
    end
    
    function Game.setGravity(gravity)
        if type(gravity) == "number" then
             pcall(tm.physics.SetGravityMultiplier, gravity)
        else
             -- Deprecated vector set
             pcall(tm.physics.SetGravity, gravity) 
        end
    end
    
    function Game.getComplexity()
        -- No getter for complexity in docs, only setter
        return 700 
    end
    
    function Game.setComplexity(value)
        pcall(tm.physics.SetBuildComplexity, value)
    end
    
    return Game
end)()

-- =================================================================
-- MODULE: PERMISSIONS LIB
-- =================================================================
_G.PermissionsLib = (function()
    local PermissionsLib = {}
    
    local PlayerLib = _G.PlayerLib
    local TimerLib = _G.TimerLib

    -- =================================================================
    -- INTERNAL STATE
    -- =================================================================

    local playerPermissions = {}
    local roles = { ["user"] = 1 } -- Default built-in role
    local defaultPermission = "user" 
    local sortedRoles = { { name = "user", level = 1 } } -- For easy promotion/demotion

    -- =================================================================
    -- HELPER FUNCTIONS
    -- =================================================================

    local function updateSortedRoles()
        sortedRoles = {}
        for name, level in pairs(roles) do
            table.insert(sortedRoles, { name = name, level = level })
        end
        table.sort(sortedRoles, function(a, b) return a.level < b.level end)
    end

    -- =================================================================
    -- PUBLIC API
    -- =================================================================

    -- Registers a role with a specific hierarchy level.
    function PermissionsLib.registerRole(options)
        if not options or not options.name or not options.level then
            Logger.error("PermissionsLib: registerRole requires a name and a numeric level.")
            return
        end
        roles[options.name] = options.level
        updateSortedRoles()
        -- If this is the first role registered (or the one with the lowest level), make it the default.
        if not defaultPermission or options.level < roles[defaultPermission] then
            defaultPermission = options.name
        end
    end

    -- Manually grants a permission level to a player for the current session.
    function PermissionsLib.grantPermission(player, level)
        if not player or not level or not roles[level] then 
            Logger.warn("PermissionsLib: Invalid player or unregistered permission level provided: " .. tostring(level))
            return 
        end
        local pid = (type(player) == "number") and player or player.id
        playerPermissions[pid] = { role = level, isTemporary = false }
        Logger.info("PermissionsLib: Set permission level for player " .. pid .. " to '" .. level .. "'.")
    end

    -- Grants a temporary permission that reverts after a duration.
    function PermissionsLib.grantTemporaryPermission(player, level, duration)
        if not player or not level or not roles[level] or not duration then
            Logger.error("PermissionsLib: grantTemporaryPermission requires a player, level, and duration.")
            return
        end
        local pid = (type(player) == "number") and player or player.id
        local originalRole = PermissionsLib.getPlayerPermission(player)
        
        playerPermissions[pid] = { role = level, isTemporary = true }
        Logger.info("PermissionsLib: Temporarily set player " .. pid .. " to '" .. level .. "' for " .. duration .. "s.")

        TimerLib.after({
            id = "temp_perm_" .. pid,
            duration = duration,
            onComplete = function()
                -- Only revert if their permission is still the temporary one.
                if playerPermissions[pid] and playerPermissions[pid].isTemporary then
                    playerPermissions[pid] = { role = originalRole, isTemporary = false }
                    Logger.info("PermissionsLib: Temporary permission for player " .. pid .. " expired. Reverted to '" .. originalRole .. "'.")
                end
            end
        })
    end

    -- Gets the current permission level of a player.
    function PermissionsLib.getPlayerPermission(player)
        local pid = (type(player) == "number") and player or player.id
        if playerPermissions[pid] then
            return playerPermissions[pid].role
        end
        return defaultPermission
    end

    -- Checks if a player has at least the required permission level.
    function PermissionsLib.hasPermission(player, requiredLevel)
        if not requiredLevel then return true end -- If no level is required, access is granted.
        
        local playerLevelStr = PermissionsLib.getPlayerPermission(player)
        
        local playerLevelNum = roles[playerLevelStr] or 0
        local requiredLevelNum = roles[requiredLevel] or 0
        
        return playerLevelNum >= requiredLevelNum
    end

    -- Helper functions for managing roles and players
    function PermissionsLib.roleExists(roleName)
        return roles[roleName] ~= nil
    end

    function PermissionsLib.getRoles()
        return sortedRoles
    end

    function PermissionsLib.getPlayersInRole(roleName)
        local playersInRole = {}
        local allPlayers = PlayerLib.getAllPlayers()
        for id, player in pairs(allPlayers) do
            if PermissionsLib.getPlayerPermission(player) == roleName then
                table.insert(playersInRole, player)
            end
        end
        return playersInRole
    end

    function PermissionsLib.promote(player)
        local currentRole = PermissionsLib.getPlayerPermission(player)
        local currentLevel = roles[currentRole] or 0
        for i, roleInfo in ipairs(sortedRoles) do
            if roleInfo.level > currentLevel then
                PermissionsLib.grantPermission(player, roleInfo.name)
                return roleInfo.name
            end
        end
        return currentRole -- Already at the highest rank
    end

    function PermissionsLib.demote(player)
        local currentRole = PermissionsLib.getPlayerPermission(player)
        local currentLevel = roles[currentRole] or 0
        for i = #sortedRoles, 1, -1 do
            local roleInfo = sortedRoles[i]
            if roleInfo.level < currentLevel then
                PermissionsLib.grantPermission(player, roleInfo.name)
                return roleInfo.name
            end
        end
        return currentRole -- Already at the lowest rank
    end

    -- =================================================================
    -- EVENT HANDLERS & INITIALIZATION
    -- =================================================================

    -- Clears a player's permissions when they leave.
    local function onPlayerLeft(player)
        playerPermissions[player.id] = nil
    end

    PlayerLib.onPlayerLeft(onPlayerLeft)

    return PermissionsLib
end)()

-- =================================================================
-- MODULE: SOUND LIB
-- =================================================================
_G.SoundLib = (function()
    local SoundLib = {}
    
    local UpdateManager = _G.UpdateManager
    if not UpdateManager then
        Logger.error("SoundLib: UpdateManager is required but was not found.")
        return {}
    end

    local trackedSounds = {}
    local soundsByHandle = {} -- O(1) reverse lookup
    local soundsToRemove = {}

    -- =================================================================
    -- PUBLIC API
    -- =================================================================

    function SoundLib.playAtPosition(options)
        if not options or not options.sound or not options.position then
            Logger.error("SoundLib: playAtPosition requires 'sound' and 'position'.")
            return
        end
        
        pcall(tm.audio.PlayAudioAtPosition, options.sound, options.position)
    end

    function SoundLib.attach(options)
        if not options or not options.handle or not options.sound or not options.id then
            Logger.error("SoundLib: attach requires a 'handle', a 'sound', and a unique 'id'.")
            return
        end

        if trackedSounds[options.id] then
            SoundLib.stop({ id = options.id })
        end

        local gameObject = options.handle.gameObject or options.handle.rootObject
        if not gameObject then return end

        pcall(tm.audio.PlayAudioAtGameobject, options.sound, gameObject)
        
        trackedSounds[options.id] = {
            handle = options.handle,
            sound = options.sound
        }
        
        -- Register in reverse index
        local handleId = options.handle.id
        if not soundsByHandle[handleId] then soundsByHandle[handleId] = {} end
        soundsByHandle[handleId][options.id] = true
    end

    function SoundLib.stop(options)
        if not options or not options.id then return end

        local soundData = trackedSounds[options.id]
        if soundData and soundData.handle and soundData.handle:Exists() then
            local gameObject = soundData.handle.gameObject or soundData.handle.rootObject
            if gameObject then
                pcall(tm.audio.StopAllAudioAtGameobject, gameObject)
            end
        end
        
        -- Clean up reverse index
        if soundData and soundData.handle then
            local handleId = soundData.handle.id
            if soundsByHandle[handleId] then
                soundsByHandle[handleId][options.id] = nil
                if next(soundsByHandle[handleId]) == nil then
                    soundsByHandle[handleId] = nil
                end
            end
        end
        
        trackedSounds[options.id] = nil
    end

    -- Stops all sounds associated with a specific SceneLib handle.
    function SoundLib.stopAllOnObject(options)
        if not options or not options.handle then return end
        local handleId = options.handle.id

        -- Optimized O(1) lookup
        if soundsByHandle[handleId] then
            for soundId, _ in pairs(soundsByHandle[handleId]) do
                table.insert(soundsToRemove, soundId)
            end
        end
    end

    function SoundLib.cleanup()
        for id, _ in pairs(trackedSounds) do
            SoundLib.stop({ id = id })
        end
        trackedSounds = {}
        soundsByHandle = {}
        tm.os.Log("SoundLib [INFO]: Cleaned up all tracked sounds.")
    end

    function SoundLib.getSoundNames()
        return tm.audio.GetAudioNames()
    end

    function SoundLib.getRandomSoundName()
        local names = tm.audio.GetAudioNames()
        if names and #names > 0 then
            return names[math.random(1, #names)]
        end
        return nil
    end

    -- Update loop to handle deferred removal of sounds
    function SoundLib.update()
        if #soundsToRemove > 0 then
            for _, id in ipairs(soundsToRemove) do
                SoundLib.stop({ id = id })
            end
            soundsToRemove = {}
        end
    end

    UpdateManager.register(SoundLib.update, 860) -- Runs after animations, before scene

    return SoundLib
end)()

-- =================================================================
-- MODULE: UI LIB
-- =================================================================
_G.UILib = (function()
    local UILib = {}
    
    local TimerLib = _G.TimerLib
    local PlayerLib = _G.PlayerLib
    local PermissionsLib = _G.PermissionsLib

    if not PlayerLib or not PermissionsLib then
        -- Should not happen in unified file
        return {}
    end
    
    local playerMenuStates = {}

    -- =================================================================
    -- HELPER FUNCTIONS
    -- =================================================================


    local function applyFilter(text, filterType, range)
        if filterType == "numbers" then
            local filteredText = (select(1, text:gsub("[^%d%.%-]", "")))
            local num = tonumber(filteredText)
            if num then
                if range then
                    num = math.max(range.min, math.min(range.max, num))
                end
                return tostring(num)
            end
            return ""
        elseif filterType == "letters" then
            return (select(1, text:gsub("[^%a%s]", "")))
        elseif filterType == "alphanumeric" then
            return (select(1, text:gsub("[^%w%s]", "")))
        end
        return text
    end

    local function playSoundForTargets(target, soundId)
        if not soundId or not tm.audio then return end
        local pids = PlayerLib.getTargetPlayers(target)
        for _, pid in ipairs(pids) do
            local playerTransform = tm.players.GetPlayerTransform(pid)
            if playerTransform then
                pcall(tm.audio.PlayAudioAtPosition, soundId, playerTransform:GetPositionWorld(), 1.0)
            end
        end
    end

    -- =================================================================
    -- UI ELEMENT FUNCTIONS
    -- =================================================================

    UILib.createLabel = function(options) for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do pcall(tm.playerUI.AddUILabel, pid, options.id, options.text) end end
    UILib.createButton = function(options) for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do pcall(tm.playerUI.AddUIButton, pid, options.id, options.text, options.callback, options.data) end end
    UILib.createInputField = function(options) for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do pcall(tm.playerUI.AddUIText, pid, options.id, options.defaultText, options.callback, options.data) end end
    UILib.updateText = function(options) for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do pcall(tm.playerUI.SetUIValue, pid, options.id, options.text) end end
    UILib.removeElement = function(options) for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do pcall(tm.playerUI.RemoveUI, pid, options.id) end end
    UILib.clearAll = function(options) for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do pcall(tm.playerUI.ClearUI, pid) end end

    -- =================================================================
    -- SUBTLE MESSAGE FUNCTIONS
    -- =================================================================

    local function internalCreateMessage(options)
        local id = options.id
        local target = options.target
        local header = truncate(options.header)
        local message = truncate(options.message)
        local messageOptions = options.options or {}

        local duration = messageOptions.duration
        local icon = messageOptions.icon
        
        if not UILib.loadedIcons then UILib.loadedIcons = {} end

        if icon and icon ~= "" and not UILib.loadedIcons[icon] then
            local isStandard = false
            if SceneLib and SceneLib.standardIcons then
                for _, std in ipairs(SceneLib.standardIcons) do if std == icon then isStandard = true; break end end
            end

            if not isStandard then
                local textureFilename = icon .. ".png"
                pcall(tm.physics.AddTexture, textureFilename, icon)
            end
            UILib.loadedIcons[icon] = true
        end

        if not UILib.trackedSubtleMessages then UILib.trackedSubtleMessages = {} end
        if id and UILib.trackedSubtleMessages[id] then UILib.removeSubtleMessage({ id = id, silent = true }) end
        
        local targets = PlayerLib.getTargetPlayers(target)
        local isForAll = (type(target) == "string" and target:lower() == "all")
        local messageData = {
            soundOnDisappear = messageOptions.soundOnDisappear
        }

        if isForAll then
            messageData.isForAll = true
            messageData.internalId = tm.playerUI.AddSubtleMessageForAllPlayers(header, message, duration, icon)
        else
            messageData.isForAll = false
            messageData.targets = targets
            messageData.internalIds = {}
            for _, pid in ipairs(targets) do
                messageData.internalIds[pid] = tm.playerUI.AddSubtleMessageForPlayer(pid, header, message, duration, icon)
            end
        end

        if id then UILib.trackedSubtleMessages[id] = messageData end
        playSoundForTargets(target, messageOptions.soundOnAppear)
    end

    UILib.createSubtleMessage = function(options)
        if not options.options then options.options = {} end
        if options.id and not options.options.duration then
            options.options.duration = 999999
        end
        internalCreateMessage(options)
    end

    UILib.updateSubtleMessageHeader = function(options)
        local msgData = UILib.trackedSubtleMessages and UILib.trackedSubtleMessages[options.id]
        if not msgData then return end

        local safeHeader = truncate(options.header)

        if msgData.isForAll then
            tm.playerUI.SubtleMessageUpdateHeaderForAll(msgData.internalId, safeHeader)
        else
            for _, pid in ipairs(msgData.targets) do
                if msgData.internalIds[pid] then tm.playerUI.SubtleMessageUpdateHeaderForPlayer(pid, msgData.internalIds[pid], safeHeader) end
            end
        end
    end

    UILib.updateSubtleMessageText = function(options)
        local msgData = UILib.trackedSubtleMessages and UILib.trackedSubtleMessages[options.id]
        if not msgData then return end

        local safeMessage = truncate(options.message)

        if msgData.isForAll then
            tm.playerUI.SubtleMessageUpdateMessageForAll(msgData.internalId, safeMessage)
        else
            for _, pid in ipairs(msgData.targets) do
                if msgData.internalIds[pid] then tm.playerUI.SubtleMessageUpdateMessageForPlayer(pid, msgData.internalIds[pid], safeMessage) end
            end
        end
    end

    UILib.removeSubtleMessage = function(options)
        local msgData = UILib.trackedSubtleMessages and UILib.trackedSubtleMessages[options.id]
        if not msgData then return end
        
        if not options.silent and msgData.soundOnDisappear then
            local target = msgData.isForAll and "all" or msgData.targets
            playSoundForTargets(target, msgData.soundOnDisappear)
        end

        if msgData.isForAll then
            tm.playerUI.RemoveSubtleMessageForAll(msgData.internalId)
        else
            for _, pid in ipairs(msgData.targets) do
                if msgData.internalIds[pid] then
                    tm.playerUI.RemoveSubtleMessageForPlayer(pid, msgData.internalIds[pid])
                end
            end
        end
        UILib.trackedSubtleMessages[options.id] = nil
    end

    -- =================================================================
    -- OPTIONS DISPLAY
    -- =================================================================
    UILib.showOptions = function(options)
        if not options or not options.target or not options.id or not options.title or not options.options then return end

        local targets = PlayerLib.getTargetPlayers(options.target)
        for _, pid in ipairs(targets) do
            local player = PlayerLib.getPlayer(pid)
            if player:getState() ~= "default" then return end -- Don't show if busy

            player:setState("choosing_option")

            UILib.clearAll({target = pid})
            UILib.createLabel({target = pid, id = "options_title", text = options.title})

            for i, opt in ipairs(options.options) do
                local callback = function()
                    player:setState("default")
                    UILib.clearAll({target = pid})
                    if type(opt.onSelect) == "function" then
                        pcall(opt.onSelect)
                    end
                end
                UILib.createButton({target = pid, id = "option_" .. i, text = opt.text, callback = callback})
            end

            if options.duration and options.duration > 0 then
                TimerLib.after({
                    duration = options.duration,
                    onComplete = function()
                        local p = PlayerLib.getPlayer(pid)
                        if p and p:isInState("choosing_option") then
                            p:setState("default")
                            UILib.clearAll({target = pid})
                            if type(options.onTimeout) == "function" then
                                pcall(options.onTimeout)
                            end
                        end
                    end
                })
            end
        end
    end


    -- =================================================================
    -- INTRUSIVE MESSAGE FUNCTIONS
    -- =================================================================

    function UILib.showIntrusiveMessage(options)
        local targets = PlayerLib.getTargetPlayers(options.target)
        local isForAll = (type(options.target) == "string" and options.target:lower() == "all")

        if isForAll then
            pcall(tm.playerUI.ShowIntrusiveMessageForAllPlayers, options.header, options.message, options.duration)
        else
            for _, pid in ipairs(targets) do
                pcall(tm.playerUI.ShowIntrusiveMessageForPlayer, pid, options.header, options.message, options.duration)
            end
        end
    end

    -- =================================================================
    -- COUNTDOWN DISPLAY FUNCTIONS
    -- =================================================================

    local function formatTimeMMSS(seconds)
        seconds = math.max(0, math.floor(seconds))
        local mins = math.floor(seconds / 60)
        local secs = seconds % 60
        return string.format("%02d:%02d", mins, secs)
    end

    UILib.createCountdown = function(options)
        if not options or not options.id or not options.target then
            tm.os.Log("UILib [ERROR]: createCountdown requires an options table with at least 'id' and 'target'.")
            return nil
        end

        local useIntrusive = options.useIntrusive or false
        if not useIntrusive and not options.header then
            tm.os.Log("UILib [ERROR]: Subtle countdowns require a 'header' in the options table.")
            return nil
        end

        local duration = options.duration or 0
        local countUp = options.countUp or false
        local timeFormatter = options.timeFormatter or formatTimeMMSS

        if not useIntrusive then
            local initialTime = countUp and 0 or duration
            local initialMessage = (options.messageFormat or "{time}"):gsub("{time}", timeFormatter(initialTime))
            UILib.createSubtleMessage({
                target = options.target, header = options.header, message = initialMessage, id = options.id,
                options = {
                    icon = options.icon,
                    soundOnAppear = options.soundOnAppear,
                    soundOnDisappear = options.soundOnDisappear
                }
            })
        end

        local onTickAndComplete = function(timerId, currentRep)
            local displayTime = countUp and currentRep or (duration - currentRep)
            local formattedMessage = (options.messageFormat or "{time}"):gsub("{time}", timeFormatter(displayTime))

            if useIntrusive then
                UILib.showIntrusiveMessage({ target = options.target, header = formattedMessage, message = "", duration = 1.5 })
            else
                UILib.updateSubtleMessageText({ id = options.id, message = formattedMessage })
            end
            
            if type(options.onTick) == "function" then pcall(options.onTick, displayTime) end
            playSoundForTargets(options.target, options.soundOnTick)

            if currentRep >= duration then
                if type(options.onComplete) == "function" then pcall(options.onComplete) end
                playSoundForTargets(options.target, options.soundOnComplete)
                
                if not useIntrusive then
                    TimerLib.after({
                        duration = 0.5,
                        onComplete = function() UILib.removeSubtleMessage({ id = options.id }) end
                    })
                end
            end
        end

        return TimerLib.every({
            id = options.id .. "_timer", interval = 1.0,
            onTick = onTickAndComplete, count = duration,
            group = options.group
        })
    end

    UILib.pauseCountdown = function(id) return TimerLib.pause(id .. "_timer") end
    UILib.resumeCountdown = function(id) return TimerLib.resume(id .. "_timer") end
    UILib.removeCountdown = function(id)
        UILib.removeSubtleMessage({ id = id })
        return TimerLib.cancel(id .. "_timer")
    end
    UILib.isCountdownPaused = function(id) return TimerLib.isPaused(id .. "_timer") end
    UILib.getCountdownRemainingTime = function(id) return TimerLib.getTimeRemaining(id .. "_timer") end

    -- =================================================================
    -- MENU & NAVIGATION LOGIC
    -- =================================================================

    local function _drawMenu(pid); end -- Forward declaration

    local menuItemDrawers = {
        label = function(pid, id, itemDef)
            UILib.createLabel({ id = id, target = pid, text = itemDef.text or "" })
        end,
        button = function(pid, id, itemDef, state)
            local onClick = itemDef.onClick
            if itemDef.opens then
                onClick = function()
                    table.insert(state.stack, itemDef.opens)
                    _drawMenu(pid, "standard")
                end
            elseif itemDef.action and string.find(itemDef.action, "back") then
                onClick = function()
                    if itemDef.action == "back_to_root" and #state.stack > 1 then
                        state.stack = { state.stack[1] }
                    else
                        table.remove(state.stack)
                    end
                    if #state.stack > 0 then _drawMenu(pid, "standard") else UILib.closeMenu({ target = pid, style = "standard" }) end
                end
            end
            UILib.createButton({ id = id, target = pid, text = itemDef.text or "Button", callback = onClick, data = itemDef.data })
        end,
        input = function(pid, id, itemDef)
            local wrappedCallback = function(data)
                local filteredValue = applyFilter(data.value, itemDef.filter, itemDef.range)
                if data.value ~= filteredValue then UILib.updateText({ id = id, target = pid, text = filteredValue }) end
                if type(itemDef.onChanged) == "function" then itemDef.onChanged({ value = filteredValue }) end
            end
            UILib.createInputField({ id = id, target = pid, defaultText = tostring(itemDef.default or ""), callback = wrappedCallback, data = itemDef.data })
        end,
        toggle = function(pid, id, itemDef)
            local initialValue = (type(itemDef.getValue) == "function" and itemDef.getValue()) or itemDef.default
            local currentIndex = 1
            for i, v in ipairs(itemDef.options) do if v == initialValue then currentIndex = i; break; end end
            
            local function updateToggleText()
                local currentValue = itemDef.options[currentIndex]
                local displayValue = tostring(currentValue)
                if type(currentValue) == "boolean" then displayValue = currentValue and "ON" or "OFF" end
                return (itemDef.text or "") .. displayValue
            end
            
            local wrappedOnClick = function()
                currentIndex = (currentIndex % #itemDef.options) + 1
                local newValue = itemDef.options[currentIndex]
                UILib.updateText({ id = id, target = pid, text = updateToggleText() })
                if type(itemDef.onToggle) == "function" then itemDef.onToggle({ value = newValue }) end
            end
            UILib.createButton({ id = id, target = pid, text = updateToggleText(), callback = wrappedOnClick })
        end
    }

    local function _drawStandardMenu(pid)
        local rootState = playerMenuStates[pid]
        if not rootState or not rootState.standard then return end
        local state = rootState.standard
        
        local definition = state.stack[#state.stack]
        UILib.clearAll({ target = pid })
        
        if definition.title then UILib.createLabel({ id = "menu_title", target = pid, text = definition.title }) end
        if not definition.items or type(definition.items) ~= "table" then
            tm.os.Log("UILib [ERROR]: Standard menu definition is missing 'items' table.")
            return
        end

        local itemCounter = 0
        for _, itemDef in ipairs(definition.items) do
            local hasPermission = not itemDef.permission or PermissionsLib.hasPermission(pid, itemDef.permission)
            
            if hasPermission and itemDef.type and menuItemDrawers[itemDef.type] then
                itemCounter = itemCounter + 1
                local itemId = itemDef.id or ("menu_item_" .. itemCounter)
                menuItemDrawers[itemDef.type](pid, itemId, itemDef, state)
            end
        end
    end

    local function _getOrBuildSubtleMenuOptions(navState, menu, isRoot)
        local allOptions = (menu.getDynamicOptions and menu.getDynamicOptions(navState.context)) or menu.options
        if not allOptions then return {} end

        local builtOptions = {}
        local player = PlayerLib.getPlayer(navState.context.playerId)

        for i = 1, #allOptions do
            local option = allOptions[i]
            if not option.permission or PermissionsLib.hasPermission(player, option.permission) then
                table.insert(builtOptions, option)
            end
        end

        if not isRoot then
            table.insert(builtOptions, { name = "Back", action = "back" })
        end
        
        return builtOptions
    end

    local function _drawSubtleMenu(pid)
        local rootState = playerMenuStates[pid]
        if not rootState or not rootState.subtle then return end
        local state = rootState.subtle

        local navState = state.stack[#state.stack]
        local menu = state.definition.menus[navState.menuKey]
        if not menu then tm.os.Log("UILib [ERROR]: Menu '" .. navState.menuKey .. "' not found."); return end

        local totalOptions = _getOrBuildSubtleMenuOptions(navState, menu, #state.stack == 1)
        
        local display = " "
        if navState.index > 0 and navState.index <= #totalOptions then
            local currentOpt = totalOptions[navState.index]
            local currentValue = ""
            if type(currentOpt.getValue) == "function" then
                local value = currentOpt.getValue(navState.context)
                if value ~= nil then
                    if currentOpt.action == "adjust" then
                        currentValue = " [" .. tostring(value) .. "]"
                    elseif currentOpt.action == "toggle" then
                        currentValue = " [" .. (value and "ON" or "OFF") .. "]"
                    end
                end
            end
            display = "→ " .. (currentOpt.name or "Unnamed") .. currentValue
        end
        
        local title = menu.title or "Menu"
        if type(menu.title) == "function" then title = menu.title(navState.context) end
        
        if state.msgId then
            tm.playerUI.SubtleMessageUpdateHeaderForPlayer(pid, state.msgId, title)
            tm.playerUI.SubtleMessageUpdateMessageForPlayer(pid, state.msgId, display)
        else
            state.msgId = tm.playerUI.AddSubtleMessageForPlayer(pid, title, display, 999999, state.definition.icon)
        end
    end

    _drawMenu = function(pid, style)
        if not playerMenuStates[pid] then return end
        
        if style == "subtle" and playerMenuStates[pid].subtle then
            _drawSubtleMenu(pid)
        elseif style == "standard" and playerMenuStates[pid].standard then
            _drawStandardMenu(pid)
        end
    end

    UILib.isOpen = function(pid, style) 
        local state = playerMenuStates[pid]
        if not state then return false end
        if style then return state[style] ~= nil end
        return (state.standard ~= nil) or (state.subtle ~= nil)
    end

    UILib.showMenu = function(options)
        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            local definition = options.definition
            local style = definition.style or "standard"
            
            -- Close existing menu of THIS style only
            if UILib.isOpen(pid, style) then UILib.closeMenu({ target = pid, style = style }) end
            
            if style == "subtle" and definition.icon and definition.icon ~= "" then
                if not UILib.loadedIcons then UILib.loadedIcons = {} end
                if not UILib.loadedIcons[definition.icon] then
                    local isStandard = false
                    if SceneLib and SceneLib.standardIcons then
                        for _, std in ipairs(SceneLib.standardIcons) do if std == definition.icon then isStandard = true; break end end
                    end

                    if not isStandard then
                        pcall(tm.physics.AddTexture, definition.icon .. ".png", definition.icon)
                    end
                    UILib.loadedIcons[definition.icon] = true
                end
            end
            
            local initialState
            if style == "subtle" then
                if not definition.root or not definition.menus then
                    tm.os.Log("UILib [ERROR]: Subtle menu requires 'root' and 'menus' tables.")
                    return
                end
                initialState = { menuKey = definition.root, index = 1, context = { playerId = pid } }
            else
                initialState = definition
            end
            
            if not playerMenuStates[pid] then playerMenuStates[pid] = {} end
            
            playerMenuStates[pid][style] = {
                stack = { initialState }, 
                style = style,
                definition = definition, 
                msgId = nil
            }
            _drawMenu(pid, style)
        end
    end

    UILib.closeMenu = function(options)
        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            local rootState = playerMenuStates[pid]
            if rootState then
                local styleToClose = options.style
                
                -- Helper to close a specific state
                local closeState = function(style)
                    local state = rootState[style]
                    if state then
                        if state.style == "subtle" and state.msgId then
                            tm.playerUI.RemoveSubtleMessageForPlayer(pid, state.msgId)
                        elseif state.style == "standard" then
                            UILib.clearAll({ target = pid })
                        end
                        rootState[style] = nil
                    end
                end

                if styleToClose then
                    closeState(styleToClose)
                else
                    -- Close everything if no style specified
                    closeState("standard")
                    closeState("subtle")
                end
                
                -- Cleanup root if empty
                if not rootState.standard and not rootState.subtle then
                    playerMenuStates[pid] = nil
                end
            end
        end
    end

    UILib.navigate = function(pid, direction)
        local rootState = playerMenuStates[pid]
        if not rootState or not rootState.subtle then return end
        local state = rootState.subtle
        
        local function activate(option, navState)
            if option.action == "back" then
                if #state.stack > 1 then table.remove(state.stack) else UILib.closeMenu({ target = pid, style = "subtle" }) end
            elseif option.action == "submenu" and option.submenu then
                local newContext = option.getContext and option.getContext(navState.context, option) or navState.context
                table.insert(state.stack, { menuKey = option.submenu, index = 1, context = newContext })
            elseif option.action == "execute" and option.onExecute then
                option.onExecute(navState.context)
            elseif option.action == "toggle" and type(option.onToggle) == "function" then
                option.onToggle(navState.context)
            end
        end

        local navState = state.stack[#state.stack]
        local menu = state.definition.menus[navState.menuKey]
        local totalOptions = _getOrBuildSubtleMenuOptions(navState, menu, #state.stack == 1)
        if #totalOptions == 0 then return end

        if direction == "up" then
            navState.index = (navState.index - 2 + #totalOptions) % #totalOptions + 1
        elseif direction == "down" then
            navState.index = (navState.index % #totalOptions) + 1
        else
            local option = totalOptions[navState.index]
            if not option then return end
            if direction == "left" or direction == "right" then
                if option.action == "adjust" and type(option.onAdjust) == "function" then
                    option.onAdjust(navState.context, (direction == "right") and 1 or -1)
                elseif option.action == "toggle" then
                    activate(option, navState)
                elseif direction == "right" then
                    activate(option, navState)
                elseif direction == "left" and #state.stack > 1 then
                    table.remove(state.stack)
                end
            end
        end
        
        if playerMenuStates[pid] then _drawMenu(pid, "subtle") end
    end

    UILib.initPlayer = function(player)
        if not player then return end
        local pid = player.playerId
        local navs = {"Up", "Down", "Left", "Right"}
        for _, nav in ipairs(navs) do
            local funcName = "UILib_Navigate"..nav
            _G[funcName] = function(p) if UILib and UILib.navigate then UILib.navigate(p, string.lower(nav)) end end
            pcall(tm.input.RegisterFunctionToKeyDownCallback, pid, funcName, string.lower(nav))
        end
    end

    UILib.onPlayerLeft = function(player)
        if not player then return end
        UILib.closeMenu({ target = player.playerId })
    end

    if tm and tm.players then
        if tm.players.OnPlayerJoined then pcall(tm.players.OnPlayerJoined.add, UILib.initPlayer) end
        if tm.players.OnPlayerLeft then pcall(tm.players.OnPlayerLeft.add, UILib.onPlayerLeft) end
    end

    -- Explicit global registration for navigation callbacks (required for string-based callbacks)
    _G.UILib_NavigateUp = function(p) if UILib and UILib.navigate then UILib.navigate(p, "up") end end
    _G.UILib_NavigateDown = function(p) if UILib and UILib.navigate then UILib.navigate(p, "down") end end
    _G.UILib_NavigateLeft = function(p) if UILib and UILib.navigate then UILib.navigate(p, "left") end end
    _G.UILib_NavigateRight = function(p) if UILib and UILib.navigate then UILib.navigate(p, "right") end end

    return UILib
end)()

-- =================================================================
-- MODULE: ANIM LIB
-- =================================================================
_G.AnimLib = (function()
    local AnimLib = {}
    
    local UpdateManager = _G.UpdateManager
    if not UpdateManager then
        Logger.error("AnimLib: UpdateManager is required but was not found. AnimLib will not function.")
        return {}
    end
    
    local animationTables = {
        active = {},      -- All per-frame animations will live here
        sequences = {},   -- Sequences are a special case (state machine)
        followers = {}    -- Followers are also a special case
    }
    
    -- =================================================================
    -- EASING FUNCTIONS
    -- =================================================================
    
    local Easing = {
        linear = function(t) return t end,
        easeInSine = function(t) return 1 - math.cos((t * math.pi) / 2) end,
        easeOutSine = function(t) return math.sin((t * math.pi) / 2) end,
        easeInOutSine = function(t) return -(math.cos(math.pi * t) - 1) / 2 end,
        easeInQuad = function(t) return t * t end,
        easeOutQuad = function(t) return 1 - (1 - t) * (1 - t) end,
        easeInOutQuad = function(t) return t < 0.5 and 2 * t * t or 1 - math.pow(-2 * t + 2, 2) / 2 end,
        easeInCubic = function(t) return t * t * t end,
        easeOutCubic = function(t) return 1 - math.pow(1 - t, 3) end,
        easeInOutCubic = function(t) return t < 0.5 and 4 * t * t * t or 1 - math.pow(-2 * t + 2, 3) / 2 end,
        easeInQuart = function(t) return t * t * t * t end,
        easeOutQuart = function(t) return 1 - math.pow(1 - t, 4) end,
        easeInOutQuart = function(t) return t < 0.5 and 8 * t * t * t * t or 1 - math.pow(-2 * t + 2, 4) / 2 end,
        easeInQuint = function(t) return t * t * t * t * t end,
        easeOutQuint = function(t) return 1 - math.pow(1 - t, 5) end,
        easeInOutQuint = function(t) return t < 0.5 and 16 * t * t * t * t * t or 1 - math.pow(-2 * t + 2, 5) / 2 end,
        easeInExpo = function(t) return t == 0 and 0 or math.pow(2, 10 * t - 10) end,
        easeOutExpo = function(t) return t == 1 and 1 or 1 - math.pow(2, -10 * t) end,
        easeInOutExpo = function(t)
            if t == 0 or t == 1 then return t end
            if t < 0.5 then return math.pow(2, 20 * t - 10) / 2
            else return (2 - math.pow(2, -20 * t + 10)) / 2 end
        end,
        easeInCirc = function(t) return 1 - math.sqrt(1 - math.pow(t, 2)) end,
        easeOutCirc = function(t) return math.sqrt(1 - math.pow(t - 1, 2)) end,
        easeInOutCirc = function(t)
            if t < 0.5 then return (1 - math.sqrt(1 - math.pow(2 * t, 2))) / 2
            else return (math.sqrt(1 - math.pow(-2 * t + 2, 2)) + 1) / 2 end
        end,
        easeInBack = function(t) local c1 = 1.70158; local c3 = c1 + 1; return c3 * t * t * t - c1 * t * t end,
        easeOutBack = function(t) local c1 = 1.70158; local c3 = c1 + 1; return 1 + c3 * math.pow(t - 1, 3) + c1 * math.pow(t - 1, 2) end,
        easeInOutBack = function(t)
            local c1 = 1.70158; local c2 = c1 * 1.525
            if t < 0.5 then return (math.pow(2 * t, 2) * ((c2 + 1) * 2 * t - c2)) / 2
            else return (math.pow(2 * t - 2, 2) * ((c2 + 1) * (t * 2 - 2) + c2) + 2) / 2 end
        end,
        easeInElastic = function(t)
            local c4 = (2 * math.pi) / 3
            if t == 0 or t == 1 then return t end
            return -math.pow(2, 10 * t - 10) * math.sin((t * 10 - 10.75) * c4)
        end,
        easeOutElastic = function(t)
            local c4 = (2 * math.pi) / 3
            if t == 0 or t == 1 then return t end
            return math.pow(2, -10 * t) * math.sin((t * 10 - 0.75) * c4) + 1
        end,
        easeInOutElastic = function(t)
            local c5 = (2 * math.pi) / 4.5
            if t == 0 or t == 1 then return t end
            if t < 0.5 then return -(math.pow(2, 20 * t - 10) * math.sin((20 * t - 11.125) * c5)) / 2
            else return (math.pow(2, -20 * t + 10) * math.sin((20 * t - 11.125) * c5)) / 2 + 1 end
        end,
        easeOutBounce = function(t)
            local n1 = 7.5625; local d1 = 2.75
            if t < 1 / d1 then return n1 * t * t
            elseif t < 2 / d1 then t = t - 1.5 / d1; return n1 * t * t + 0.75
            elseif t < 2.5 / d1 then t = t - 2.25 / d1; return n1 * t * t + 0.9375
            else t = t - 2.625 / d1; return n1 * t * t + 0.984375 end
        end
    }
    Easing.easeInBounce = function(t) return 1 - Easing.easeOutBounce(1 - t) end
    Easing.easeInOutBounce = function(t)
        if t < 0.5 then return (1 - Easing.easeOutBounce(1 - 2 * t)) / 2
        else return (1 + Easing.easeOutBounce(2 * t - 1)) / 2 end
    end
    
    function AnimLib.registerEasing(name, func)
        if type(name) == "string" and type(func) == "function" then
            Easing[name] = func
            Logger.info("AnimLib: Registered custom easing function '" .. name .. "'.")
        else
            Logger.error("AnimLib: Invalid arguments for registerEasing.")
        end
    end
    
    -- =================================================================
    -- ANIMATION UPDATERS & HELPERS
    -- =================================================================
    
    local animationUpdaters = {}
    local function getEasedPingPong(progress, easeFunc)
        local pingPongProgress = progress * 2
        if pingPongProgress > 1 then
            pingPongProgress = 2 - pingPongProgress
        end
        return easeFunc(pingPongProgress)
    end
    
    local function createLookRotation(forward, up)
        up = up or tm.vector3.Up()
        local f = (forward:Magnitude() > 0.0001) and tm.vector3.op_Division(forward, forward:Magnitude()) or tm.vector3.Create(0,0,1)
        local r = tm.vector3.op_Division(tm.vector3.Cross(up, f), tm.vector3.Cross(up, f):Magnitude())
        local u = tm.vector3.Cross(f, r)
        local m00, m01, m02 = r.x, r.y, r.z
        local m10, m11, m12 = u.x, u.y, u.z
        local m20, m21, m22 = f.x, f.y, f.z
        
        local tr = m00 + m11 + m22
        local qw, qx, qy, qz
        if tr > 0 then
            local S = math.sqrt(tr+1.0) * 2;
            qw, qx, qy, qz = 0.25 * S, (m21 - m12) / S, (m02 - m20) / S, (m10 - m01) / S
        elseif (m00 > m11) and (m00 > m22) then
            local S = math.sqrt(1.0 + m00 - m11 - m22) * 2
            qw, qx, qy, qz = (m21 - m12) / S, 0.25 * S, (m10 + m01) / S, (m02 + m20) / S
        elseif m11 > m22 then
            local S = math.sqrt(1.0 + m11 - m00 - m22) * 2
            qw, qx, qy, qz = (m02 - m20) / S, (m10 + m01) / S, 0.25 * S, (m21 + m12) / S
        else
            local S = math.sqrt(1.0 + m22 - m00 - m11) * 2
            qw, qx, qy, qz = (m10 - m01) / S, (m02 + m20) / S, (m21 + m12) / S, 0.25 * S
        end
        return tm.quaternion.Create(qx, qy, qz, qw)
    end
    
    local function getTargetPosition(target, objectPos)
        if type(target) == "table" and target.Exists and pcall(function() return target:Exists() end) and target.GetTransform then
            return target:GetTransform():GetPositionWorld()
        elseif type(target) == "table" and target.x and target.y and target.z then
            return target
        else
            local closestPlayerPos, minDstSq = nil, math.huge
            for _, player in ipairs(tm.players.CurrentPlayers()) do
                local playerTransform = tm.players.GetPlayerTransform(player.playerId)
                if playerTransform then
                    local playerPos = playerTransform:GetPositionWorld()
                    local dstSq = tm.vector3.op_Subtraction(playerPos, objectPos):Magnitude() ^ 2
                    if dstSq < minDstSq then minDstSq, closestPlayerPos = dstSq, playerPos end
                end
            end
            return closestPlayerPos
        end
    end
    
    animationUpdaters.tween = function(anim, dt)
        anim.elapsed = (anim.elapsed or 0) + (dt * anim.speedMultiplier)
        local progress = anim.elapsed / anim.duration
    
        if progress >= 1 then
            if anim.pingpong and not anim.isReversing then
                anim.isReversing = true
                if anim.targetPosition then anim.originalPosition, anim.targetPosition = anim.targetPosition, anim.originalPosition end
                if anim.targetRotation then anim.originalRotation, anim.targetRotation = anim.targetRotation, anim.originalRotation end
                if anim.targetScale then anim.originalScale, anim.targetScale = anim.targetScale, anim.originalScale end
                anim.elapsed = 0
                return false
            elseif anim.loop then
                if anim.onLoop then pcall(anim.onLoop, anim.handle) end
                anim.elapsed = 0
                return false
            else
                if anim.isReversing then
                    if anim.absoluteOriginalRotation then anim.handle:setRotation(anim.absoluteOriginalRotation) end
                    if anim.absoluteOriginalScale then anim.handle:setScale(anim.absoluteOriginalScale) end
                    if anim.absoluteOriginalPosition then anim.handle:setPosition(anim.absoluteOriginalPosition) end
                else
                    if anim.targetRotation then anim.handle:setRotation(anim.targetRotation) end
                    if anim.targetScale then anim.handle:setScale(anim.targetScale) end
                    if anim.targetPosition then anim.handle:setPosition(anim.targetPosition) end
                end
                return true
            end
        else
            local easedProgress = anim.easeFunc(progress)
            if anim.targetRotation then anim.handle:setRotation(tm.quaternion.Slerp(anim.originalRotation, anim.targetRotation, easedProgress)) end
            
            local basePosition = anim.originalPosition
            if anim.targetPosition then basePosition = tm.vector3.Lerp(anim.originalPosition, anim.targetPosition, easedProgress) end
            
            if anim.targetScale then
                local currentScale = tm.vector3.Lerp(anim.originalScale, anim.targetScale, easedProgress)
                anim.handle:setScale(currentScale)
                if anim.scaleDirection and anim.scaleDirection:Magnitude() > 0.0001 and basePosition then
                    local scaleChange = tm.vector3.op_Subtraction(currentScale, anim.absoluteOriginalScale)
                    local pivot = anim.pivotPoint or tm.vector3.Create(0,0,0)
                    local normalizedPivot = tm.vector3.op_Multiply(tm.vector3.op_Addition(pivot, tm.vector3.Create(1,1,1)), 0.5)
                    local offsetFactorX = 0; if anim.scaleDirection.x == -1 then offsetFactorX = normalizedPivot.x elseif anim.scaleDirection.x == 1 then offsetFactorX = -(1 - normalizedPivot.x) end
                    local offsetFactorY = 0; if anim.scaleDirection.y == -1 then offsetFactorY = normalizedPivot.y elseif anim.scaleDirection.y == 1 then offsetFactorY = -(1 - normalizedPivot.y) end
                    local offsetFactorZ = 0; if anim.scaleDirection.z == -1 then offsetFactorZ = normalizedPivot.z elseif anim.scaleDirection.z == 1 then offsetFactorZ = -(1 - normalizedPivot.z) end
                    local localOffset = tm.vector3.Create(offsetFactorX * scaleChange.x, offsetFactorY * scaleChange.y, offsetFactorZ * scaleChange.z)
                    anim.handle:setPosition(tm.vector3.op_Addition(basePosition, anim.handle:GetTransform():TransformDirection(localOffset)))
                elseif basePosition then
                    anim.handle:setPosition(basePosition)
                end
            elseif basePosition then
                anim.handle:setPosition(basePosition)
            end
        end
        return false
    end
    
    animationUpdaters.path = function(anim, dt)
        if anim.isPausing then
            if tm.os.GetTime() >= anim.pauseEndTime then
                anim.isPausing, anim.elapsed = false, 0
            end
            return false
        end
    
        local from, to = anim.currentFrame, anim.isReversing and anim.currentFrame - 1 or anim.currentFrame + 1
        local startFrame, targetFrame = anim.frames[from], anim.frames[to]
        if not targetFrame then
            if anim.loop then
                if anim.onLoop then pcall(anim.onLoop, anim.handle) end
                if anim.pingpong then anim.isReversing = not anim.isReversing; anim.currentFrame = anim.isReversing and #anim.frames -1 or 2
                else anim.currentFrame = 1 end
                anim.elapsed = 0
            elseif anim.pingpong and not anim.isReversing then anim.isReversing = true; anim.currentFrame = #anim.frames -1; anim.elapsed = 0
            else return true end
        else
            local duration = targetFrame.duration or 1.0
            anim.elapsed = (anim.elapsed or 0) + (dt * anim.speedMultiplier)
            local progress = duration > 0 and anim.elapsed / duration or 1
            if progress >= 1 then
                if targetFrame.position then anim.handle:setPosition(targetFrame.position) end
                if targetFrame.rotation then anim.handle:setRotation(targetFrame.rotation) end
                anim.currentFrame, anim.elapsed = to, 0
                if targetFrame.pause and targetFrame.pause > 0 then anim.isPausing, anim.pauseEndTime = true, tm.os.GetTime() + targetFrame.pause end
            else
                local eased = anim.easeFunc(progress)
                if startFrame.position and targetFrame.position then anim.handle:setPosition(tm.vector3.Lerp(startFrame.position, targetFrame.position, eased)) end
                if startFrame.rotation and targetFrame.rotation then anim.handle:setRotation(tm.quaternion.Slerp(startFrame.rotation, targetFrame.rotation, eased)) end
            end
        end
        return false
    end
    
    local function updateSimpleLoopingAnim(anim, dt, onUpdateApply)
        anim.elapsed = (anim.elapsed or 0) + (dt * anim.speedMultiplier)
        if not anim.loop and anim.elapsed >= anim.duration then
            onUpdateApply(anim, 1.0, true)
            return true
        else
            local progress = (anim.elapsed % anim.duration) / anim.duration
            local easedValue = getEasedPingPong(progress, anim.easeFunc)
            onUpdateApply(anim, easedValue, false)
    
            if anim.loop and anim.elapsed >= anim.duration then
                if anim.onLoop then pcall(anim.onLoop, anim.handle) end
                anim.elapsed = anim.elapsed - anim.duration
            end
            return false
        end
    end
    
    animationUpdaters.pulse = function(anim, dt)
        return updateSimpleLoopingAnim(anim, dt, function(a, eased, isFinal)
            if isFinal then a.handle:setScale(a.originalScale); return end
            local newScale = a.minScale + (a.maxScale - a.minScale) * eased
            a.handle:setScale(tm.vector3.Create(newScale, newScale, newScale))
        end)
    end
    
    animationUpdaters.bob = function(anim, dt)
        return updateSimpleLoopingAnim(anim, dt, function(a, eased, isFinal)
            if isFinal then if not a.handle.parent then a.handle:setPosition(a.originalPosition) end; return end
            local newPos = tm.vector3.op_Addition(a.originalPosition, tm.vector3.op_Multiply(tm.vector3.Up(), (eased * 2 - 1) * a.height))
            if a.handle.parent then a.handle.localPosition = newPos else a.handle:setPosition(newPos) end
        end)
    end
    
    animationUpdaters.sway = function(anim, dt)
        return updateSimpleLoopingAnim(anim, dt, function(a, eased, isFinal)
            if isFinal then if not a.handle.parent then a.handle:setPosition(a.originalPosition) end; return end
            local newPos = tm.vector3.op_Addition(a.originalPosition, tm.vector3.op_Multiply(a.axis, (eased * 2 - 1) * a.distance))
            if a.handle.parent then a.handle.localPosition = newPos else a.handle:setPosition(newPos) end
        end)
    end
    
    animationUpdaters.spin = function(anim, dt)
        local rotAmount = anim.baseSpeed * anim.speedMultiplier * dt
        local deltaRotation = tm.quaternion.Create(rotAmount, anim.axis)
        if anim.handle.parent then 
            anim.handle.localRotation = deltaRotation:Multiply(anim.handle.localRotation)
        else 
            anim.handle:setRotation(deltaRotation:Multiply(anim.handle:GetTransform():GetRotationWorld())) 
        end
        return false
    end
    
    animationUpdaters.shake = function(anim, dt)
        anim.elapsed = (anim.elapsed or 0) + (dt * anim.speedMultiplier)
        if anim.elapsed < anim.duration then
            local offset = tm.vector3.op_Multiply(tm.vector3.Create(math.random()*2-1, math.random()*2-1, math.random()*2-1), anim.intensity)
            anim.handle:setPosition(tm.vector3.op_Addition(anim.originalPosition, offset))
        else
            anim.handle:setPosition(anim.originalPosition)
            if anim.loop then
                anim.elapsed = 0
                if anim.onLoop then pcall(anim.onLoop, anim.handle) end
                return false
            else
                return true
            end
        end
        return false
    end
    
    animationUpdaters.orbit = function(anim, dt)
        local elapsed = (anim.elapsed or 0) + (dt * anim.speedMultiplier)
        anim.elapsed = elapsed
        
        local angle = (elapsed * anim.baseSpeed) % 360
        local rad = math.rad(angle)
        local cosA = math.cos(rad)
        local sinA = math.sin(rad)
        
        -- Scalar math to avoid vector allocation for orbitOffset
        local offX, offY, offZ = cosA * anim.radius, 0, sinA * anim.radius
        
        -- Avoid op_Addition
        local cx, cy, cz = anim.center.x, anim.center.y, anim.center.z
        local newPos = tm.vector3.Create(cx + offX, cy + offY, cz + offZ)
        
        anim.handle:setPosition(newPos)
    
        if anim.faceCenter then
            -- Scalar direction calculation
            local dirX, dirY, dirZ = cx - newPos.x, cy - newPos.y, cz - newPos.z
            local magSq = dirX*dirX + dirY*dirY + dirZ*dirZ
            if magSq > 0.000001 then
                -- Re-use existing createLookRotation helper but pass vector (wrapper handles it)
                -- Or manually construct quaternion if critical, but CreateLookRotation is native content generally.
                -- For now, reusing helper is fine as we saved 2 vector allocs above.
                local lookDir = tm.vector3.Create(dirX, dirY, dirZ) 
                anim.handle:setRotation(createLookRotation(lookDir, anim.axis)) 
            end
        end
        return false
    end
    
    animationUpdaters.flicker = function(anim, dt)
        anim.elapsed = (anim.elapsed or 0) + (dt * anim.speedMultiplier)
        if not anim.loop and anim.elapsed >= anim.duration then
            anim.handle.gameObject:SetIsVisible(anim.originalVisibility)
            return true
        else
            anim.handle.gameObject:SetIsVisible(math.fmod(anim.elapsed * anim.baseSpeed, 2.0) < 1.0)
            if anim.loop and anim.elapsed >= anim.duration then
                if anim.onLoop then pcall(anim.onLoop, anim.handle) end
                anim.elapsed = anim.elapsed - anim.duration
            end
        end
        return false
    end
    
    animationUpdaters.lookAt = function(anim, dt)
        local handlePos = anim.handle:GetTransform():GetPositionWorld()
        local hx, hy, hz = handlePos.x, handlePos.y, handlePos.z
        local targetPos = getTargetPosition(anim.target, handlePos)
        
        if targetPos then
            local tx, ty, tz = targetPos.x, targetPos.y, targetPos.z
            local dirX, dirY, dirZ = tx - hx, ty - hy, tz - hz
            local magSq = dirX*dirX + dirY*dirY + dirZ*dirZ
            
            if magSq > 0.000001 then
               -- Local Rotation logic (Parented)
               if anim.handle.parent then
                    
                    local groupTransform = anim.handle.parent:GetTransform()
                    local groupRot = groupTransform:GetRotationWorld()
                    local invGroupRot = tm.quaternion.Create(-groupRot.x, -groupRot.y, -groupRot.z, groupRot.w)
                    local worldDir = tm.vector3.Create(dirX, dirY, dirZ) -- Re-alloc needed for Quat math as is
                    local p_dir = tm.quaternion.Create(worldDir.x, worldDir.y, worldDir.z, 0)
                    local rotated_p = invGroupRot:Multiply(p_dir):Multiply(groupRot)
                    local localDir = tm.vector3.Create(rotated_p.x, rotated_p.y, rotated_p.z)
                    
                    if localDir:Magnitude() > 0.001 then
                        local yawAngle = math.atan2(localDir.x, localDir.z) * (180 / math.pi)
                        local adjacent = math.sqrt(localDir.x^2 + localDir.z^2) -- Sqrt of sum squares (Magnitude horizontal)
                        local pitchAngle = -math.atan2(localDir.y, adjacent) * (180 / math.pi)
                        local targetLocalRot = tm.quaternion.Create(0, yawAngle, 0):Multiply(tm.quaternion.Create(pitchAngle, 0, 0))
                        
                        if anim.smoothing and anim.smoothing > 0 then
                            local t = 1 - math.exp(-10 * dt / anim.smoothing)
                            anim.handle.localRotation = tm.quaternion.Slerp(anim.handle.localRotation, targetLocalRot, t)
                        else
                            anim.handle.localRotation = targetLocalRot
                        end
                    end
               else
                   -- World Rotation case
                   local dirVec = tm.vector3.Create(dirX, dirY, dirZ)
                   local targetWorldRot = createLookRotation(dirVec, anim.up)
                   
                   if anim.smoothing and anim.smoothing > 0 then
                       local t = 1 - math.exp(-10 * dt / anim.smoothing)
                       anim.handle:setRotation(tm.quaternion.Slerp(anim.handle:GetTransform():GetRotationWorld(), targetWorldRot, t))
                   else
                       anim.handle:setRotation(targetWorldRot)
                   end
               end
            end
        end
        return false
    end
    
    -- =================================================================
    -- CORE ANIMATION LOGIC
    -- =================================================================
    
    function _createAnimation(options, type, setupFunc)
        if not options or not options.handle then return end
        local handle = options.handle
        
        if not options.isFromSequence and not options.isCrossfade then
            AnimLib.stop({ handle = handle })
        end
    
        local anim = {
            startTime = tm.os.GetTime(),
            isPaused = false,
            pauseTime = 0,
            speedMultiplier = 1.0,
            elapsed = 0,
            type = type
        }
        for k, v in pairs(options) do
            anim[k] = v
        end
    
        anim.easeFunc = Easing[anim.easeType] or Easing.linear
    
        if setupFunc then
            pcall(setupFunc, handle, anim)
        end
        
        animationTables.active[handle.id] = anim
    
        if anim.onStart and type(anim.onStart) == "function" then
            pcall(anim.onStart, handle)
        end
    end
    
    -- =================================================================
    -- PUBLIC ANIMATION API
    -- =================================================================
    
    function AnimLib.sequence(options)
        if not options or not options.handle or not options.animations or #options.animations == 0 then return end
        local handle = options.handle
        AnimLib.stop({ handle = handle })
        
        animationTables.sequences[handle.id] = {
            handle = handle,
            animations = options.animations,
            currentIndex = 1,
            direction = 1,
            loop = options.loop or false,
            pingpong = options.pingpong or false,
            onStart = options.onStart,
            onLoop = options.onLoop,
            onComplete = options.onComplete,
            isPaused = false,
            isWaiting = false,
            pauseTime = 0,
            hasStarted = false
        }
    end
    
    function AnimLib.sequenceGroup(options)
        if not options or not options.groupHandle or not options.animations or #options.animations == 0 then return end
        
        local group = options.groupHandle
        local delay = options.delay or 0.1
        local objects = group:getObjects()
    
        for i, objectHandle in ipairs(objects) do
            local seqOptions = {
                handle = objectHandle,
                animations = options.animations,
                loop = options.loop,
                pingpong = options.pingpong
            }
    
            if delay > 0 then
                TimerLib.after({
                    duration = i * delay,
                    onComplete = function()
                        if objectHandle and objectHandle:Exists() then
                            AnimLib.sequence(seqOptions)
                        end
                    end
                })
            else
                AnimLib.sequence(seqOptions)
            end
        end
    end
    
    function AnimLib.crossfade(options)
        if not options or not options.handle or not options.to or not AnimLib[options.to] then return end
        
        local handle = options.handle
        local currentTransform = handle:GetTransform()
        local startPos = currentTransform:GetPositionWorld()
        local startRot = currentTransform:GetRotationWorld()
        local startScale = currentTransform:GetScale()
    
        AnimLib.stop({ handle = handle })
        
        local toOptions = options.toOptions or {}
        toOptions.handle = handle
        
        local endPos, endRot, endScale = startPos, startRot, startScale
        
        local tempAnimState = {}
        local setupFunc = function(h, a)
            if a.originalPosition then endPos = a.originalPosition end
            if a.originalRotation then endRot = a.originalRotation end
            if a.originalScale then endScale = a.originalScale end
        end
        _createAnimation(toOptions, options.to, setupFunc)
        
        animationTables.active[handle.id] = nil
        
        AnimLib.tween({
            handle = handle,
            targetPosition = endPos,
            targetRotation = endRot,
            targetScale = endScale,
            duration = options.duration or 0.5,
            isCrossfade = true,
            onComplete = function()
                toOptions.isCrossfade = false
                AnimLib[options.to](toOptions)
            end
        })
    end
    
    function AnimLib.tween(options)
        if not options or not options.handle or (not options.targetPosition and not options.targetRotation and not options.targetScale) then return end
        
        local setup = function(h, a)
            local hTransform = h:GetTransform()
            if a.targetPosition then
                a.originalPosition = hTransform:GetPositionWorld()
                if a.isRelative then a.targetPosition = tm.vector3.op_Addition(a.originalPosition, a.targetPosition) end
                a.absoluteOriginalPosition = a.originalPosition
            end
            if a.targetRotation then
                a.originalRotation = hTransform:GetRotationWorld()
                if a.isRelative then a.targetRotation = a.targetRotation:Multiply(a.originalRotation) end
                a.absoluteOriginalRotation = a.originalRotation
            end
            if a.targetScale then
                a.originalScale = hTransform:GetScale()
                if a.isRelative then a.targetScale = tm.vector3.op_Addition(a.originalScale, a.targetScale) end
                a.absoluteOriginalScale = a.originalScale
                if a.scaleDirection and not a.originalPosition then
                    a.originalPosition = hTransform:GetPositionWorld()
                    a.absoluteOriginalPosition = a.originalPosition
                end
            end
        end
    
        if type(options.targetScale) == "number" then
            local s = options.targetScale
            options.targetScale = tm.vector3.Create(s, s, s)
        end
        
        options.duration = options.duration or 1.0
        options.easeType = options.easeType or "linear"
    
        _createAnimation(options, "tween", setup)
    end
    
    function AnimLib.path(options)
        if not options or not options.handle or not options.frames or #options.frames < 2 then return end
        
        options.currentFrame = 1
        options.isReversing = false
        options.isPausing = false
        options.pauseEndTime = 0
        options.easeType = options.easeType or "linear"
        
        local setup = function(h, a)
            local startFrame = a.frames[1]
            if startFrame.position then h:setPosition(startFrame.position) end
            if startFrame.rotation then h:setRotation(startFrame.rotation) end
        end
    
        _createAnimation(options, "path", setup)
    end
    
    function AnimLib.follow(options)
        if not options or not options.handle or not options.target then return end
        _createAnimation(options, "follow")
    end
    
    function AnimLib.pulse(options)
        if not options or not options.handle then return end
        local speed = options.speed or 0.6
        if speed == 0 then tm.os.Log("AnimLib [WARNING]: Pulse speed cannot be zero."); return end
    
        options.baseSpeed = speed
        options.minScale = options.minScale or 0.8
        options.maxScale = options.maxScale or 1.2
        options.easeType = options.easeType or "easeInOutSine"
        options.duration = 1 / speed
        
        local setup = function(h, a) a.originalScale = h:GetTransform():GetScale() end
        _createAnimation(options, "pulse", setup)
    end
    
    function AnimLib.spin(options)
        if not options or not options.handle then return end
        local speed = options.speed or 90
        if speed == 0 then tm.os.Log("AnimLib [WARNING]: Spin speed cannot be zero."); return end
        
        options.baseSpeed = speed
        options.axis = options.axis or tm.vector3.Up()
        options.isReversing = false
        
        local setup = function(h, a)
            a.originalRotation = h:GetTransform():GetRotationWorld()
            if h.parent then a.originalLocalRotation = h.localRotation end
        end
        _createAnimation(options, "spin", setup)
    end
    
    function AnimLib.shake(options)
        if not options or not options.handle then return end
        options.intensity = options.intensity or 0.1
        options.duration = options.duration or 1.0
        local setup = function(h, a) a.originalPosition = h:GetTransform():GetPositionWorld() end
        _createAnimation(options, "shake", setup)
    end
    
    function AnimLib.orbit(options)
        if not options or not options.handle then return end
        local speed = options.speed or 30
        if speed == 0 then tm.os.Log("AnimLib [WARNING]: Orbit speed cannot be zero."); return end
    
        options.baseSpeed = speed
        options.center = options.center or options.handle:GetTransform():GetPositionWorld()
        options.radius = options.radius or 5
        options.axis = options.axis or tm.vector3.Up()
        options.isReversing = false
    
        local setup = function(h, a)
            local firstPointOffset = tm.vector3.Create(a.radius, 0, 0)
            local firstPoint = tm.vector3.op_Addition(a.center, firstPointOffset)
            h:setPosition(firstPoint)
            a.originalPosition = firstPoint
            a.originalRotation = h:GetTransform():GetRotationWorld()
        end
        _createAnimation(options, "orbit", setup)
    end
    
    function AnimLib.bob(options)
        if not options or not options.handle then return end
        local speed = options.speed or 0.6
        if speed == 0 then tm.os.Log("AnimLib [WARNING]: Bob speed cannot be zero."); return end
    
        options.baseSpeed = speed
        options.height = options.height or 0.5
        options.easeType = options.easeType or "easeInOutSine"
        options.duration = 1 / speed
        local setup = function(h, a) a.originalPosition = h:GetTransform():GetPositionWorld() end
        _createAnimation(options, "bob", setup)
    end
    
    function AnimLib.flicker(options)
        if not options or not options.handle then return end
        local speed = options.speed or 10
        if speed == 0 then tm.os.Log("AnimLib [WARNING]: Flicker speed cannot be zero."); return end
        
        options.baseSpeed = speed
        options.duration = 2 / speed
        local setup = function(h, a) a.originalVisibility = h.gameObject:GetIsVisible() end
        _createAnimation(options, "flicker", setup)
    end
    
    function AnimLib.sway(options)
        if not options or not options.handle then return end
        local speed = options.speed or 0.6
        if speed == 0 then tm.os.Log("AnimLib [WARNING]: Sway speed cannot be zero."); return end
        
        options.baseSpeed = speed
        options.distance = options.distance or 1.0
        options.axis = options.axis or tm.vector3.Right()
        options.easeType = options.easeType or "easeInOutSine"
        options.duration = 1 / speed
        local setup = function(h, a) a.originalPosition = h:GetTransform():GetPositionWorld() end
        _createAnimation(options, "sway", setup)
    end
    
    function AnimLib.lookAt(options)
        if not options or not options.handle then return end
        _createAnimation(options, "lookAt")
    end
    
    function AnimLib.stop(options)
        if not options or not options.handle or not options.handle:Exists() then return end
        local handle = options.handle
        local id = handle.id
    
        local anim = animationTables.active[id]
        if anim then
            if anim.absoluteOriginalPosition then handle:setPosition(anim.absoluteOriginalPosition) elseif anim.originalPosition then handle:setPosition(anim.originalPosition) end
            if anim.absoluteOriginalRotation then handle:setRotation(anim.absoluteOriginalRotation) elseif anim.originalRotation then handle:setRotation(anim.originalRotation) end
            if anim.absoluteOriginalScale then handle:setScale(anim.absoluteOriginalScale) elseif anim.originalScale then handle:setScale(anim.originalScale) end
            if anim.originalVisibility ~= nil then handle.gameObject:SetIsVisible(anim.originalVisibility) end
            animationTables.active[id] = nil
        end
        
        animationTables.sequences[id] = nil
        animationTables.followers[id] = nil
    end
    
    function AnimLib.pause(options)
        if not options or not options.handle then return end
        local id = options.handle.id
        for _, animTable in pairs(animationTables) do
            local anim = animTable[id]
            if anim and not anim.isPaused then
                anim.isPaused = true
                anim.pauseTime = tm.os.GetTime()
                return
            end
        end
    end
    
    function AnimLib.resume(options)
        if not options or not options.handle then return end
        local id = options.handle.id
        for _, animTable in pairs(animationTables) do
            local anim = animTable[id]
            if anim and anim.isPaused then
                anim.isPaused = false
                local pausedDuration = tm.os.GetTime() - anim.pauseTime
                anim.startTime = (anim.startTime or 0) + pausedDuration
                if anim.pauseEndTime then
                    anim.pauseEndTime = anim.pauseEndTime + pausedDuration
                end
                return
            end
        end
    end
    
    function AnimLib.cleanup()
        for name, animTable in pairs(animationTables) do
            for k in pairs(animTable) do
                animTable[k] = nil
            end
        end
        tm.os.Log("AnimLib [INFO]: Cleaned up all animations.")
    end
    
    function AnimLib.setSpeed(options)
        if not options or not options.handle or not options.speed then return end
        local id = options.handle.id
        local anim = animationTables.active[id]
        if anim then
            anim.speedMultiplier = options.speed
        end
    end
    
    -- =================================================================
    -- CORE UPDATE LOOP
    -- =================================================================
    
    function AnimLib.update()
        local dt = tm.os.GetModDeltaTime()
        if dt == 0 then return end
        local keysToRemove = {}
    
        for id, seq in pairs(animationTables.sequences) do
            if not (seq.isPaused or seq.isWaiting) then
                if not seq.hasStarted then
                    seq.hasStarted = true
                    if seq.onStart then pcall(seq.onStart, seq.handle) end
                else
                    seq.currentIndex = seq.currentIndex + seq.direction
                end
                
                local isFinished = false
                if seq.direction == 1 and seq.currentIndex > #seq.animations then
                    if seq.loop then
                        if seq.onLoop then pcall(seq.onLoop, seq.handle) end
                        if seq.pingpong then
                            seq.direction = -1
                            seq.currentIndex = #seq.animations - 1
                        else
                            seq.currentIndex = 1
                        end
                    elseif seq.pingpong and not seq.isReversing then
                        seq.direction = -1
                        seq.isReversing = true
                        seq.currentIndex = #seq.animations - 1
                    else
                        isFinished = true
                    end
                elseif seq.direction == -1 and seq.currentIndex < 1 then
                    if seq.loop and seq.pingpong then
                        if seq.onLoop then pcall(seq.onLoop, seq.handle) end
                        seq.direction = 1
                        seq.isReversing = false
                        seq.currentIndex = 2
                    else
                        isFinished = true
                    end
                end
                
                if isFinished then
                    if seq.onComplete then pcall(seq.onComplete, seq.handle) end
                    table.insert(keysToRemove, {tbl = animationTables.sequences, key = id})
                else
                    local animToPlay = seq.animations[seq.currentIndex]
                    if animToPlay then
                        local newOptions = {}
                        for k,v in pairs(animToPlay.options or {}) do newOptions[k] = v end
                        
                        newOptions.isFromSequence = true
                        
                        local originalOnComplete = newOptions.onComplete
                        newOptions.onComplete = function(h)
                            if originalOnComplete then pcall(originalOnComplete, h) end
                            local currentSeq = animationTables.sequences[id]
                            if currentSeq then currentSeq.isWaiting = false end
                        end
    
                        local animDef = { type = animToPlay.type, options = newOptions }
                        
                        if seq.handle and seq.handle:Exists() then
                            local animOptions = animDef.options or {}
                            animOptions.handle = seq.handle
                            animOptions.loop = false 
                            
                            if AnimLib[animDef.type] then
                                AnimLib[animDef.type](animOptions)
                                seq.isWaiting = true
                            end
                        end
                    else
                        table.insert(keysToRemove, {tbl = animationTables.sequences, key = id})
                    end
                end
            end
        end
    
        for id, anim in pairs(animationTables.active) do
            if not anim.isPaused and anim.handle and anim.handle:Exists() then
                local updater = animationUpdaters[anim.type]
                if updater then
                    local isFinished = updater(anim, dt)
                    if isFinished then
                        if anim.onComplete then pcall(anim.onComplete, anim.handle) end
                        table.insert(keysToRemove, {tbl = animationTables.active, key = id})
                    end
                end
            else
                table.insert(keysToRemove, {tbl = animationTables.active, key = id})
            end
        end
    
        for id, follower in pairs(animationTables.followers) do
            if not follower.isPaused then
                if not follower.target or not follower.target:Exists() then
                    table.insert(keysToRemove, {tbl = animationTables.followers, key = id})
                else
                    local t = 1 - math.exp(-(follower.smoothing or 10) * dt)
                    local targetTransform = follower.target:GetTransform()
                    local targetPosition = targetTransform:GetPositionWorld()
                    local targetRotation = targetTransform:GetRotationWorld()
                    -- Optimized math to avoid op_Addition and Lerp allocations
                    local offsetVec = targetTransform:TransformDirection(follower.positionOffset or tm.vector3.Create(0,0,0))
                    local desX, desY, desZ = targetPosition.x + offsetVec.x, targetPosition.y + offsetVec.y, targetPosition.z + offsetVec.z
                    
                    -- Manual Lerp
                    local curPos = follower.handle:GetTransform():GetPositionWorld()
                    local lerpX = curPos.x + (desX - curPos.x) * t
                    local lerpY = curPos.y + (desY - curPos.y) * t
                    local lerpZ = curPos.z + (desZ - curPos.z) * t
                    
                    follower.handle:setPosition(tm.vector3.Create(lerpX, lerpY, lerpZ))
                    if follower.followRotation then
                        local desiredRot = targetRotation:Multiply(follower.rotationOffset or tm.quaternion.Create(0,0,0,1))
                        follower.handle:setRotation(tm.quaternion.Slerp(follower.handle:GetTransform():GetRotationWorld(), desiredRot, t))
                    end
                end
            end
        end
    
        for _, toRemove in ipairs(keysToRemove) do
            if toRemove.tbl[toRemove.key] then
                toRemove.tbl[toRemove.key] = nil
            end
        end
    end
    
    UpdateManager.register(AnimLib.update, 850)
    
    return AnimLib
end)()

-- =================================================================
-- MODULE: INPUT LIB
-- =================================================================
_G.InputLib = (function()
    local InputLib = {}
    
    local PlayerLib = _G.PlayerLib
    local PermissionsLib = _G.PermissionsLib
    local SceneLib = _G.SceneLib

    -- tm.os.SetModTargetDeltaTime(1/60) -- Global setting

    InputLib.registeredCallbacks = { keyDown = {}, keyUp = {}, mouseDown = {}, objectClicked = {} }
    InputLib.playerContexts = {}
    InputLib.dispatchersCreated = { keyDown = {}, keyUp = {} }

    local chatCommands = {}

    -- =================================================================
    -- HELPER FUNCTIONS
    -- =================================================================

    local function parseVectorString(str)
        if type(str) ~= "string" then return nil end
        -- Robust match for "x, y, z" inside optional parens, ignoring spaces
        local x, y, z = str:match("([-?%d%.]+)%s*,%s*([-?%d%.]+)%s*,%s*([-?%d%.]+)")
        if x and y and z then
            return tm.vector3.Create(tonumber(x), tonumber(y), tonumber(z))
        end
        return nil
    end

    local function _fireCallbacks(callbacksForEvent, playerId, ...)
        if not callbacksForEvent then return end
        
        local context = InputLib.playerContexts[playerId] or "default"
        
        if callbacksForEvent[context] then
            for _, cb in ipairs(callbacksForEvent[context]) do pcall(cb, playerId, ...) end
        end
        if callbacksForEvent["global"] then
            for _, cb in ipairs(callbacksForEvent["global"]) do pcall(cb, playerId, ...) end
        end
    end

    -- =================================================================
    -- GLOBAL DISPATCHERS
    -- =================================================================

    function InputLib_MouseDown_Dispatcher(arg1, arg2)
        local playerId, posStr
        
        -- Handle Userdata (C# event object), Table (Lua event), or Direct Args
        local argType = type(arg1)
        if (argType == "table" or argType == "userdata") and arg1.playerId then
            playerId = arg1.playerId
            posStr = arg1.value
        else
            playerId = arg1
            posStr = arg2
        end

        if not InputLib or not playerId then return end
        
        -- The game API gives us the 3D world position of the click directly.
        local clickPosition = parseVectorString(posStr)
        if not clickPosition then return end

        if _G.EventLib then _G.EventLib.emit("MouseDown", playerId, clickPosition) end

        -- Fire the basic onMouseDown event with the 3D position
        _fireCallbacks(InputLib.registeredCallbacks.mouseDown, playerId, clickPosition)

        local currentSceneLib = SceneLib or _G.SceneLib
        if currentSceneLib then
            local clickedObject = currentSceneLib.findClosestEntity(clickPosition, { maxDistance = 6.0 }) -- Find object within 2 meters of the click
            if clickedObject and clickedObject.type == "object" then
                 if _G.EventLib then _G.EventLib.emit("ObjectClicked", playerId, clickedObject) end
                 _fireCallbacks(InputLib.registeredCallbacks.objectClicked, playerId, clickedObject)
            end
        end
    end

    local function createAndRegisterKeyDispatcher(eventType, keyName)
        local dispatcherId = eventType .. "_" .. keyName
        if InputLib.dispatchersCreated[dispatcherId] then return end

        local funcName = "InputLib_Dispatcher_" .. dispatcherId
        _G[funcName] = function(playerId)
            if not InputLib then return end
            
            if _G.EventLib then _G.EventLib.emit(eventType, playerId, keyName) end
            
            local callbacksForKey = InputLib.registeredCallbacks[eventType][keyName]
            _fireCallbacks(callbacksForKey, playerId, keyName)
        end
        InputLib.dispatchersCreated[dispatcherId] = true
    end

    function InputLib_OnChat_Dispatcher(senderName, message)
        
        local playerObject = PlayerLib.getPlayerByName(senderName)

        if not playerObject then
            Logger.warn("InputLib: Received chat message from an unknown player: " .. senderName)
            return
        end

        if _G.EventLib then _G.EventLib.emit("Chat", playerObject, message) end

        local callbacks = InputLib.registeredCallbacks.chat and InputLib.registeredCallbacks.chat["global"]
        _fireCallbacks(callbacks, playerObject.playerId, message)

        if string.sub(message, 1, 1) == "/" then
            local args = {}
            for arg in string.gmatch(message, "[^%s]+") do
                table.insert(args, arg)
            end
            
            if #args == 0 then return end
            local commandName = string.lower(string.sub(args[1], 2))
            
            local command = chatCommands[commandName]
            if command then
                if PermissionsLib.hasPermission(playerObject, command.permission) then
                    pcall(command.callback, playerObject, args)
                else
                    InputLib.messagePlayer({ player = playerObject, message = "You do not have permission to use this command.", color = tm.color.Red() })
                end
            else
                InputLib.messagePlayer({ player = playerObject, message = "Unknown command: " .. commandName, color = tm.color.Red() })
            end
        end
    end


    -- =================================================================
    -- UTILITY & EVENT HANDLERS
    -- =================================================================

    function InputLib.onPlayerJoined(player)
        local pid = player.playerId
        InputLib.playerContexts[pid] = "default"

        for keyName, _ in pairs(InputLib.registeredCallbacks.keyDown) do
            pcall(tm.input.RegisterFunctionToKeyDownCallback, pid, "InputLib_Dispatcher_keyDown_" .. keyName, keyName)
        end
        for keyName, _ in pairs(InputLib.registeredCallbacks.keyUp) do
            pcall(tm.input.RegisterFunctionToKeyUpCallback, pid, "InputLib_Dispatcher_keyUp_" .. keyName, keyName)
        end

        pcall(tm.playerUI.RegisterMouseDownPositionCallback, pid, InputLib_MouseDown_Dispatcher)
    end

    function InputLib.onPlayerLeft(player)
        local pid = player.playerId
        pcall(tm.playerUI.DeregisterMouseDownPositionCallback, pid, InputLib_MouseDown_Dispatcher)
        InputLib.playerContexts[pid] = nil
    end

    -- =================================================================
    -- PUBLIC API
    -- =================================================================

    function InputLib.setContext(options)
        if not options or not options.target then return end
        local P_Lib = PlayerLib or _G.PlayerLib
        if not P_Lib then return end
        
        for _, pid in ipairs(P_Lib.getTargetPlayers(options.target)) do
            InputLib.playerContexts[pid] = options.context or "default"
        end
    end

    function InputLib.getContext(playerId)
        return InputLib.playerContexts[playerId] or "default"
    end

    function InputLib.onKeyDown(keyName, callback, options)
        options = options or {}
        local context = options.context or "global"
        createAndRegisterKeyDispatcher("keyDown", keyName)
        
        InputLib.registeredCallbacks.keyDown[keyName] = InputLib.registeredCallbacks.keyDown[keyName] or {}
        InputLib.registeredCallbacks.keyDown[keyName][context] = InputLib.registeredCallbacks.keyDown[keyName][context] or {}
        table.insert(InputLib.registeredCallbacks.keyDown[keyName][context], callback)
    end

    function InputLib.onKeyUp(keyName, callback, options)
        options = options or {}
        local context = options.context or "global"
        createAndRegisterKeyDispatcher("keyUp", keyName)

        InputLib.registeredCallbacks.keyUp[keyName] = InputLib.registeredCallbacks.keyUp[keyName] or {}
        InputLib.registeredCallbacks.keyUp[keyName][context] = InputLib.registeredCallbacks.keyUp[keyName][context] or {}
        table.insert(InputLib.registeredCallbacks.keyUp[keyName][context], callback)
    end

    function InputLib.onMouseDown(callback, options)
        options = options or {}
        local context = options.context or "global"
        
        InputLib.registeredCallbacks.mouseDown[context] = InputLib.registeredCallbacks.mouseDown[context] or {}
        table.insert(InputLib.registeredCallbacks.mouseDown[context], callback)
    end

    function InputLib.whileKeyHeld(keyName, callback, options)
        options = options or {}
        -- Start a timer on Down loop
        InputLib.onKeyDown(keyName, function(playerId)
             if not InputLib.heldKeys then InputLib.heldKeys = {} end
             local k = playerId .. "_" .. keyName
             if InputLib.heldKeys[k] then TimerLib.cancel(InputLib.heldKeys[k]) end
             
             InputLib.heldKeys[k] = TimerLib.every({
                 id = "keyheld_" .. k,
                 interval = 0.0, -- Run evey frame/tick
                 onTick = function() pcall(callback, playerId) end
             })
        end, options)

        -- Stop timer on Up
        InputLib.onKeyUp(keyName, function(playerId)
             if not InputLib.heldKeys then return end
             local k = playerId .. "_" .. keyName
             if InputLib.heldKeys[k] then 
                TimerLib.cancel(InputLib.heldKeys[k]) 
                InputLib.heldKeys[k] = nil
             end
        end, options)
    end

    function InputLib.onChat(callback)
        InputLib.registeredCallbacks.chat = InputLib.registeredCallbacks.chat or {}
        InputLib.registeredCallbacks.chat["global"] = InputLib.registeredCallbacks.chat["global"] or {}
        table.insert(InputLib.registeredCallbacks.chat["global"], callback)
    end

    function InputLib.onObjectClicked(callback, options)
        options = options or {}
        local context = options.context or "global"
        
        InputLib.registeredCallbacks.objectClicked[context] = InputLib.registeredCallbacks.objectClicked[context] or {}
        table.insert(InputLib.registeredCallbacks.objectClicked[context], callback)
    end

    function InputLib.clearContext(contextName)
        if not contextName then return end
        for eventType, eventMaps in pairs(InputLib.registeredCallbacks) do
            if eventType == "mouseDown" or eventType == "objectClicked" then
                eventMaps[contextName] = nil
            else
                for _, contexts in pairs(eventMaps) do
                    contexts[contextName] = nil
                end
            end
        end
        tm.os.Log("InputLib [INFO]: Cleared all callbacks for context '" .. contextName .. "'")
    end

    -- =================================================================
    -- CHAT & COMMAND API
    -- =================================================================

    function InputLib.registerCommand(options)
        if not options or not options.name or not options.callback then
            tm.os.Log("InputLib [ERROR]: registerCommand requires a name and a callback.")
            return
        end
        
        local name = string.lower(options.name)
        local commandData = {
            name = name,
            callback = options.callback,
            permission = options.permission or "user",
            description = options.description or "No description provided."
        }
        chatCommands[name] = commandData

        if options.aliases then
            for _, alias in ipairs(options.aliases) do
                chatCommands[string.lower(alias)] = commandData
            end
        end
    end

    function InputLib.getAvailableCommands(playerObject)
        local availableCmds = {}
        local processedCmds = {}
        local Perm_Lib = PermissionsLib or _G.PermissionsLib
        
        if not Perm_Lib then return {} end
        
        for _, cmd in pairs(chatCommands) do
            if not processedCmds[cmd] then
                if Perm_Lib.hasPermission(playerObject, cmd.permission) then
                    table.insert(availableCmds, cmd)
                end
                processedCmds[cmd] = true
            end
        end
        
        return availableCmds
    end


    function InputLib.messagePlayer(options)
        if not options or not options.player or not options.message then return end
        local sender = "Server"
        local color = options.color or tm.color.White()
        local messageForOne = string.format("[%s]: %s", sender, options.message)
        pcall(tm.playerUI.SendChatMessage, options.player.name, messageForOne, color)
    end

    function InputLib.broadcast(message, color)
        local sender = "Server"
        color = color or tm.color.Yellow()
        pcall(tm.playerUI.SendChatMessage, sender, message, color)
    end

    function InputLib.serverMessage(message)
        tm.os.Log("[Server Message] " .. message)
        InputLib.broadcast(message)
    end

    -- =================================================================
    -- INITIALIZATION
    -- =================================================================

    if tm and tm.players then
        if tm.players.OnPlayerJoined then pcall(tm.players.OnPlayerJoined.add, InputLib.onPlayerJoined) end
        if tm.players.OnPlayerLeft then pcall(tm.players.OnPlayerLeft.add, InputLib.onPlayerLeft) end
        
        -- Initialize for players already in the game (essential for mod reloads)
        local currentPlayers = tm.players.CurrentPlayers()
        for _, p in ipairs(currentPlayers) do
            InputLib.onPlayerJoined(p)
        end
    end

    if tm and tm.playerUI and tm.playerUI.OnChatMessage then
        pcall(tm.playerUI.OnChatMessage.add, InputLib_OnChat_Dispatcher)
    end
    
    -- Explicit global registration for dispatchers if needed (handled dynamically but explicit global hook is good)
    _G.InputLib_MouseDown_Dispatcher = InputLib_MouseDown_Dispatcher
    _G.InputLib_OnChat_Dispatcher = InputLib_OnChat_Dispatcher

    return InputLib
end)()

-- =================================================================
-- MODULE: CAMERA LIB
-- =================================================================
_G.CameraLib = (function()
    local CameraLib = {}
    -- tm.os.SetModTargetDeltaTime(1/60) -- Global setting

    local TimerLib = _G.TimerLib
    local json = _G.json
    local UpdateManager = _G.UpdateManager
    local PlayerLib = _G.PlayerLib

    CameraLib._playerStates = {}
    local loadedPaths = {}
    local TIMER_GROUP = "CameraLibTimers"

    --------------------------------------------------------------------------------
    -- UTILITY FUNCTIONS
    --------------------------------------------------------------------------------

    local function QuaternionToDirectionVector(q)
        local x = 2 * (q.x * q.z + q.w * q.y)
        local y = 2 * (q.y * q.z - q.w * q.x)
        local z = 1 - 2 * (q.x * q.x + q.y * q.y)
        return tm.vector3.Create(x, y, z)
    end

    local function LookRotation(direction)
        if direction:Magnitude() < 0.0001 then
            return tm.quaternion.Create(0, 0, 0, 1)
        end
        
        local normalizedDir = tm.vector3.op_Division(direction, direction:Magnitude())
        local yaw = math.atan2(normalizedDir.x, normalizedDir.z)
        local pitch = math.asin(-normalizedDir.y)
        
        return tm.quaternion.Create(tm.vector3.Create(math.deg(pitch), math.deg(yaw), 0))
    end

    local function isGameObject(obj)
        if obj == nil then return false end
        local success, result = pcall(function() return type(obj.GetTransform) end)
        return success and result == "function"
    end

    local function DeserializePath(data)
        local path = { keyframes = {} }
        if not data then return path end
        for _, frameData in ipairs(data) do
            table.insert(path.keyframes, {
                time = frameData.time,
                position = tm.vector3.Create(frameData.pos.x, frameData.pos.y, frameData.pos.z),
                rotation = tm.quaternion.Create(frameData.rot.x, frameData.rot.y, frameData.rot.z, frameData.rot.w)
            })
        end
        path.totalDuration = #path.keyframes > 0 and path.keyframes[#path.keyframes].time or 0
        return path
    end

    local function LoadPath(filename, fromDynamic)
        if not fromDynamic and loadedPaths[filename] then return loadedPaths[filename] end
        local jsonString = fromDynamic and tm.os.ReadAllText_Dynamic(filename) or tm.os.ReadAllText_Static(filename)
        if not jsonString or jsonString == "" then
            tm.os.Log("CameraLib [ERROR]: Cinematic file not found or is empty: " .. filename)
            return nil
        end
        
        local success, data = pcall(json.parse, jsonString)
        if not success or not data then
            tm.os.Log("CameraLib [ERROR]: Failed to parse JSON in file: " .. filename)
            return nil
        end
        local path = DeserializePath(data)
        if not fromDynamic then loadedPaths[filename] = path end
        return path
    end

    local function InitializeCameraAtState(playerId, startPos, startRot)
        local state = CameraLib._playerStates[playerId]
        if not state or not state.isActive then
            state = { isActive = true, isPaused = false }
            CameraLib._playerStates[playerId] = state
            pcall(tm.players.AddCamera, playerId, startPos, QuaternionToDirectionVector(startRot))
            pcall(tm.players.ActivateCamera, playerId, 0.5)
        end
        
        state.currentPosition = startPos
        state.currentRotation = startRot
        pcall(tm.players.SetCameraPosition, playerId, state.currentPosition)
        pcall(tm.players.SetCameraRotation, playerId, QuaternionToDirectionVector(state.currentRotation))
        return state
    end

    local function InitializeCameraFromPlayerView(playerId)
        local playerTransform = tm.players.GetPlayerTransform(playerId)
        if not playerTransform then return nil end
        return InitializeCameraAtState(playerId, playerTransform:GetPositionWorld(), playerTransform:GetRotationWorld())
    end

    --------------------------------------------------------------------------------
    -- PUBLIC API
    --------------------------------------------------------------------------------

    function CameraLib.release(options)
        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            local state = CameraLib._playerStates[pid]
            if state and state.isActive then
                pcall(tm.players.DeactivateCamera, pid, 0.5)
                
                if state.timerId then
                    TimerLib.cancel(state.timerId)
                end

                CameraLib._playerStates[pid] = nil
            end
        end
    end

    function CameraLib.isPlaying(playerId)
        local state = CameraLib._playerStates[playerId]
        return state and state.isActive and state.mode == "cinematic"
    end

    function CameraLib.pause(options)
        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            local state = CameraLib._playerStates[pid]
            if state and state.isActive and not state.isPaused then
                state.isPaused = true
                if state.timerId then
                    TimerLib.pause(state.timerId)
                end
            end
        end
    end

    function CameraLib.resume(options)
        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            local state = CameraLib._playerStates[pid]
            if state and state.isActive and state.isPaused then
                state.isPaused = false
                if state.timerId then
                    TimerLib.resume(state.timerId)
                end
            end
        end
    end

    function CameraLib.playCinematic(options)
        local filesToLoad = (type(options.filenames) == "string") and {options.filenames} or options.filenames
        if type(filesToLoad) ~= "table" then return end

        local pathQueue = {}
        for _, filename in ipairs(filesToLoad) do
            local path = LoadPath(filename, options.fromDynamic or false)
            if path and #path.keyframes >= 2 then table.insert(pathQueue, path) end
        end
        if #pathQueue == 0 then return end
        
        local startPath = pathQueue[1]
        local startPos, startRot = startPath.keyframes[1].position, startPath.keyframes[1].rotation

        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            CameraLib.release({ target = pid })
            local state = InitializeCameraAtState(pid, startPos, startRot)
            state.mode = "cinematic"
            state.cinematic = {
                pathQueue = pathQueue, currentPathIndex = 1, path = startPath,
                speed = options.speed or 0.6, loop = options.loop or false, pingpong = options.pingpong or false,
                smoothing = options.smoothing or 0.05, direction = 1, currentTime = 0.0,
                onFinished = options.onFinished,
                currentSegmentIndex = 1
            }
        end
    end

    function CameraLib.tweenTo(options)
        if options.destination == nil then return end
        local isObject = isGameObject(options.destination)
        if isObject and not options.destination:Exists() then return end
        
        local targetPos = isObject and options.destination:GetTransform():GetPositionWorld() or options.destination
        local positionOffset = options.positionOffset or tm.vector3.Create(0, 5, -10)
        local endPosition = tm.vector3.op_Addition(targetPos, positionOffset)
        local lookAtOffset = options.lookAtOffset or tm.vector3.Create(0, 1, 0)
        local lookAtPoint = tm.vector3.op_Addition(targetPos, lookAtOffset)
        local direction = tm.vector3.op_Subtraction(lookAtPoint, endPosition)
        local endRotation = LookRotation(direction)
        local duration = options.duration or 3.0

        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            CameraLib.release({ target = pid })
            local state = InitializeCameraFromPlayerView(pid)
            if state then
                state.mode = "tween"
                state.tween = {
                    startTime = tm.os.GetTime(),
                    duration = duration,
                    startPos = state.currentPosition,
                    startRot = state.currentRotation,
                    endPos = endPosition,
                    endRot = endRotation
                }
                state.timerId = TimerLib.after({
                    duration = duration,
                    onComplete = function() CameraLib.release({ target = pid }) end,
                    group = TIMER_GROUP,
                    useGameTime = true
                })
            end
        end
    end

    function CameraLib.follow(options)
        if options.followTarget == nil then return end
        local isObject = isGameObject(options.followTarget)
        if isObject and not options.followTarget:Exists() then return end

        local targetTransform = isObject and options.followTarget:GetTransform() or nil
        local targetPos = isObject and targetTransform:GetPositionWorld() or options.followTarget
        local positionOffset = options.positionOffset or tm.vector3.Create(0, 3, -8)
        local startPos = tm.vector3.op_Addition(targetPos, isObject and targetTransform:TransformDirection(positionOffset) or positionOffset)
        local lookAtOffset = options.lookAtOffset or tm.vector3.Create(0, 1, 0)
        local lookAtPoint = tm.vector3.op_Addition(targetPos, lookAtOffset)
        local startRot = LookRotation(tm.vector3.op_Subtraction(lookAtPoint, startPos))

        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            CameraLib.release({ target = pid })
            local state = InitializeCameraAtState(pid, startPos, startRot)
            state.mode = "follow"
            state.follow = {
                target = options.followTarget, isObject = isObject, positionOffset = positionOffset,
                lookAtOffset = lookAtOffset, smoothing = options.smoothing or 1.0
            }
        end
    end

    function CameraLib.lookAt(options)
        if options.lookAtTarget == nil then return end
        local isObject = isGameObject(options.lookAtTarget)
        if isObject and not options.lookAtTarget:Exists() then return end

        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            CameraLib.release({ target = pid })
            local targetPos = isObject and options.lookAtTarget:GetTransform():GetPositionWorld() or options.lookAtTarget
            local state = CameraLib._playerStates[pid]
            local startPos
            if options.positionOffset then
               startPos = tm.vector3.op_Addition(targetPos, options.positionOffset)
            else
               startPos = (state and state.isActive) and state.currentPosition or tm.players.GetPlayerTransform(pid):GetPositionWorld()
            end
            local lookAtOffset = options.lookAtOffset or tm.vector3.Create(0, 0, 0)
            local lookAtPoint = tm.vector3.op_Addition(targetPos, lookAtOffset)
            local startRot = LookRotation(tm.vector3.op_Subtraction(lookAtPoint, startPos))
            
            state = InitializeCameraAtState(pid, startPos, startRot)
            state.mode = "lookAt"
            state.lookAt = {
                target = options.lookAtTarget, isObject = isObject, positionOffset = options.positionOffset,
                lookAtOffset = lookAtOffset, smoothing = options.smoothing or 1.0
            }
            
            if options.duration and options.duration > 0 then
                state.timerId = TimerLib.after({
                    duration = options.duration,
                    onComplete = function() CameraLib.release({ target = pid }) end,
                    group = TIMER_GROUP,
                    useGameTime = true
                })
            end
        end
    end

    function CameraLib.orbit(options)
        if options.orbitTarget == nil then return end
        local isObject = isGameObject(options.orbitTarget)
        if isObject and not options.orbitTarget:Exists() then return end

        local targetPos = isObject and options.orbitTarget:GetTransform():GetPositionWorld() or options.orbitTarget
        local yaw, pitch, distance = options.startYaw or 0.0, options.fixedPitch or 10.0, options.distance or 15.0
        local yawRad, pitchRad = math.rad(yaw), math.rad(pitch)
        local x, y, z = distance * math.cos(pitchRad) * math.sin(yawRad), distance * math.sin(pitchRad), distance * math.cos(pitchRad) * math.cos(yawRad)
        local startPos = tm.vector3.op_Addition(targetPos, tm.vector3.Create(x, y, z))
        local lookAtOffset = options.lookAtOffset or tm.vector3.Create(0, 1, 0)
        local lookAtPoint = tm.vector3.op_Addition(targetPos, lookAtOffset)
        local startRot = LookRotation(tm.vector3.op_Subtraction(lookAtPoint, startPos))

        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            CameraLib.release({ target = pid })
            local state = InitializeCameraAtState(pid, startPos, startRot)
            state.mode = "orbit"
            state.orbit = {
                target = options.orbitTarget, isObject = isObject, distance = distance,
                yawSpeed = options.yawSpeed or 30.0, fixedPitch = pitch, currentYaw = yaw,
                lookAtOffset = lookAtOffset, smoothing = options.smoothing or 1.0
            }

            if options.duration and options.duration > 0 then
                state.timerId = TimerLib.after({
                    duration = options.duration,
                    onComplete = function() CameraLib.release({ target = pid }) end,
                    group = TIMER_GROUP,
                    useGameTime = true
                })
            end
        end
    end

    --------------------------------------------------------------------------------
    -- UPDATE LOOP (STATE MACHINE)
    --------------------------------------------------------------------------------

    function CameraLib.update()
        local delta = tm.os.GetModDeltaTime()
        if next(CameraLib._playerStates) == nil then return end

        for pid, state in pairs(CameraLib._playerStates) do
            if state and state.isActive and not state.isPaused then
                local mode = state.mode
                local shouldProcess = true
                local targetObject
                
                if mode == "follow" then targetObject = state.follow.target
                elseif mode == "lookAt" and state.lookAt.isObject then targetObject = state.lookAt.target
                elseif mode == "orbit" and state.orbit.isObject then targetObject = state.orbit.target
                end
                
                if targetObject and not targetObject:Exists() then shouldProcess = false end

                if shouldProcess then
                    if mode == "tween" then
                        local tween = state.tween
                        local progress = (tm.os.GetTime() - tween.startTime) / tween.duration
                        local alpha = math.min(progress, 1.0)
                        alpha = 1 - (1 - alpha) ^ 3
                        state.currentPosition = tm.vector3.Lerp(tween.startPos, tween.endPos, alpha)
                        state.currentRotation = tm.quaternion.Slerp(tween.startRot, tween.endRot, alpha)

                        local follow = state.follow
                        local targetPos = follow.target:GetTransform():GetPositionWorld()
                        local rotatedOffset = follow.isObject and follow.target:GetTransform():TransformDirection(follow.positionOffset) or follow.positionOffset
                        
                        -- Manual addition for desiredPos
                        local desiredX = targetPos.x + rotatedOffset.x
                        local desiredY = targetPos.y + rotatedOffset.y
                        local desiredZ = targetPos.z + rotatedOffset.z
                        local desiredPos = tm.vector3.Create(desiredX, desiredY, desiredZ)
                        
                        local t = 1 - math.exp(-follow.smoothing * delta * 10)
                        state.currentPosition = tm.vector3.Lerp(state.currentPosition, desiredPos, t)
                        
                        -- Manual subtraction for direction
                        local lp = follow.lookAtOffset
                        local lookAtX, lookAtY, lookAtZ = targetPos.x + lp.x, targetPos.y + lp.y, targetPos.z + lp.z
                        local dx, dy, dz = lookAtX - state.currentPosition.x, lookAtY - state.currentPosition.y, lookAtZ - state.currentPosition.z
                        
                        -- LookRotation needs a Vector3, so we create one for the direction
                        local directionToTarget = tm.vector3.Create(dx, dy, dz)
                        local desiredRot = LookRotation(directionToTarget)
                        state.currentRotation = tm.quaternion.Slerp(state.currentRotation, desiredRot, t)

                    elseif mode == "lookAt" then
                        local lookAt = state.lookAt
                        local targetPos = lookAt.isObject and lookAt.target:GetTransform():GetPositionWorld() or lookAt.target
                        local lp = lookAt.lookAtOffset
                        local targetX, targetY, targetZ = targetPos.x + lp.x, targetPos.y + lp.y, targetPos.z + lp.z
                        local dx, dy, dz = targetX - state.currentPosition.x, targetY - state.currentPosition.y, targetZ - state.currentPosition.z
                        local dist = math.sqrt(dx*dx + dy*dy + dz*dz)
                        local t = 1 - math.exp(-(lookAt.smoothing or 10.0) * delta * 10)
                        
                        if dist > 0.0001 then
                             -- Resetting rotation from direction manually is complex, using tm.vector3.Create here is acceptable as it's once per frame per camera
                             local directionToTarget = tm.vector3.Create(dx, dy, dz)
                             local desiredRot = LookRotation(directionToTarget)
                             state.currentRotation = tm.quaternion.Slerp(state.currentRotation, desiredRot, t)
                        end
                        if lookAt.positionOffset then
                            local po = lookAt.positionOffset
                            -- Actually targetX includes lp, so targetPos is targetX-lp.
                            -- Let's just use component math from scratch to be safe and clear.
                            local tx, ty, tz = targetPos.x, targetPos.y, targetPos.z
                            local desX, desY, desZ = tx + po.x, ty + po.y, tz + po.z
                            local desPos = tm.vector3.Create(desX, desY, desZ)
                            state.currentPosition = tm.vector3.Lerp(state.currentPosition, desPos, t)
                        end
                        
                        local lookAtPoint = tm.vector3.Create(targetX, targetY, targetZ)
                        local directionToTarget = tm.vector3.op_Subtraction(lookAtPoint, state.currentPosition)
                        local desiredRot = LookRotation(directionToTarget)
                        state.currentRotation = tm.quaternion.Slerp(state.currentRotation, desiredRot, t)

                    elseif mode == "orbit" then
                        local orbit = state.orbit
                        local targetPos = orbit.isObject and orbit.target:GetTransform():GetPositionWorld() or orbit.target
                        orbit.currentYaw = orbit.currentYaw + (orbit.yawSpeed * delta)
                        local yawRad, pitchRad = math.rad(orbit.currentYaw), math.rad(orbit.fixedPitch)
                        local x = orbit.distance * math.cos(pitchRad) * math.sin(yawRad)
                        local y = orbit.distance * math.sin(pitchRad)
                        local z = orbit.distance * math.cos(pitchRad) * math.cos(yawRad)
                        local offsetPos = tm.vector3.Create(x, y, z)
                        local desiredPos = tm.vector3.op_Addition(targetPos, offsetPos)
                        local t = 1 - math.exp(-orbit.smoothing * delta * 10)
                        state.currentPosition = tm.vector3.Lerp(state.currentPosition, desiredPos, t)
                        local lookAtPoint = tm.vector3.op_Addition(targetPos, orbit.lookAtOffset)
                        local directionToTarget = tm.vector3.op_Subtraction(lookAtPoint, state.currentPosition)
                        local desiredRot = LookRotation(directionToTarget)
                        state.currentRotation = tm.quaternion.Slerp(state.currentRotation, desiredRot, t)

                    elseif mode == "cinematic" then
                        local cinematic = state.cinematic
                        cinematic.currentTime = cinematic.currentTime + (delta * cinematic.speed * cinematic.direction)
                        local currentPath = cinematic.path
                        local isAtEnd = cinematic.currentTime >= currentPath.totalDuration
                        local isAtStart = cinematic.currentTime <= 0
                        local clipFinished = (cinematic.direction == 1 and isAtEnd) or (cinematic.direction == -1 and isAtStart)
                        
                        if clipFinished then
                            if cinematic.direction == 1 then
                                local nextIndex = cinematic.currentPathIndex + 1
                                if nextIndex > #cinematic.pathQueue then
                                    if cinematic.pingpong then cinematic.direction = -1; cinematic.currentTime = currentPath.totalDuration
                                    elseif cinematic.loop then cinematic.currentPathIndex = 1; cinematic.path = cinematic.pathQueue[1]; cinematic.currentTime = 0; cinematic.currentSegmentIndex = 1;
                                    else if type(cinematic.onFinished) == "function" then cinematic.onFinished(pid) end; CameraLib.release({ target = pid }) end
                                else
                                    cinematic.currentPathIndex = nextIndex; cinematic.path = cinematic.pathQueue[nextIndex]; cinematic.currentTime = 0; cinematic.currentSegmentIndex = 1;
                                end
                            else -- direction is -1
                                local prevIndex = cinematic.currentPathIndex - 1
                                if prevIndex < 1 then
                                    if cinematic.loop then cinematic.direction = 1; cinematic.currentTime = 0; cinematic.currentSegmentIndex = 1;
                                    else if type(cinematic.onFinished) == "function" then cinematic.onFinished(pid) end; CameraLib.release({ target = pid }) end
                                else
                                    cinematic.currentPathIndex = prevIndex; cinematic.path = cinematic.pathQueue[prevIndex]; cinematic.currentTime = cinematic.path.totalDuration; cinematic.currentSegmentIndex = #cinematic.path.keyframes - 1;
                                end
                            end
                        end

                        if state.isActive then
                            local pathToUpdate = cinematic.path
                            while cinematic.currentSegmentIndex < #pathToUpdate.keyframes and pathToUpdate.keyframes[cinematic.currentSegmentIndex + 1].time < cinematic.currentTime do
                                cinematic.currentSegmentIndex = cinematic.currentSegmentIndex + 1
                            end
                            
                            local startFrame = pathToUpdate.keyframes[cinematic.currentSegmentIndex]
                            local endFrame = pathToUpdate.keyframes[cinematic.currentSegmentIndex + 1] or startFrame
                            local alpha = (endFrame.time - startFrame.time > 0) and (cinematic.currentTime - startFrame.time) / (endFrame.time - startFrame.time) or 0
                            local targetPosition = tm.vector3.Lerp(startFrame.position, endFrame.position, alpha)
                            local targetRotation = tm.quaternion.Slerp(startFrame.rotation, endFrame.rotation, alpha)
                            state.currentPosition = tm.vector3.Lerp(state.currentPosition, targetPosition, cinematic.smoothing)
                            state.currentRotation = tm.quaternion.Slerp(state.currentRotation, targetRotation, cinematic.smoothing)
                        end
                    end

                    pcall(tm.players.SetCameraPosition, pid, state.currentPosition)
                    pcall(tm.players.SetCameraRotation, pid, QuaternionToDirectionVector(state.currentRotation))
                else
                    CameraLib.release({ target = pid })
                end
            end
        end
    end

    UpdateManager.register(CameraLib.update)
    
    return CameraLib
end)()

-- =================================================================
-- MODULE: SCENE LIB
-- =================================================================
_G.SceneLib = (function()
    local SceneLib = {}
    -- tm.os.SetModTargetDeltaTime(1/60) -- Global setting

    local UpdateManager = _G.UpdateManager
    
    local TimerLib = _G.TimerLib
    local PlayerLib = _G.PlayerLib
    local AnimLib = _G.AnimLib
    local SoundLib = _G.SoundLib

    -- =================================================================
    -- CONFIGURATION
    -- =================================================================
    SceneLib.config = {
        gridCellSize = 60
    }

    -- =================================================================
    -- INTERNAL STATE & EVENTS
    -- =================================================================
    SceneLib.trackedObjects = {}
    SceneLib.trackedGroups = {}
    SceneLib.trackedTriggers = {}
    SceneLib.virtualTriggers = {}
    SceneLib.trackedPlayers = {}
    SceneLib.pools = {}
    local idCounter = 0

    function SceneLib.setWind(velocity)
        pcall(tm.world.SetGlobalWind, velocity)
    end

    function SceneLib.getWind(position)
        return tm.world.GetWindVelocityAtPosition(position)
    end

    function SceneLib.getMapName()
        return tm.physics.GetMapName()
    end
    
    function SceneLib.getAllSpawnableNames()
        return tm.physics.SpawnableNames()
    end
    
    function SceneLib.getRandomPrefabName()
        local names = tm.physics.SpawnableNames()
        if names and #names > 0 then
            return names[math.random(1, #names)]
        end
        return nil
    end

    -- Event handling system
    SceneLib.events = {
        onObjectSpawned = {},
        onObjectDespawned = {}
    }

    local function fireEvent(eventName, ...)
        if _G.EventLib then _G.EventLib.emit(eventName, ...) end
        if SceneLib.events[eventName] then
            for _, callback in ipairs(SceneLib.events[eventName]) do
                pcall(callback, ...)
            end
        end
    end

    local Grid = {
        cells = {},
        cellSize = SceneLib.config.gridCellSize,
        inverseCellSize = 1 / SceneLib.config.gridCellSize
    }

    local function generateId(prefix)
        idCounter = idCounter + 1
        return (prefix or "obj") .. "_" .. tostring(tm.os.GetTime()) .. "_" .. tostring(idCounter)
    end

    local updateNodeRecursively
    local _updateEntityInGrid

    -- =================================================================
    -- REFACTORED SHARED FUNCTIONS
    -- =================================================================

    local function _setNodePosition(node, position)
        if node.parent then
            local parentTransform = node.parent:GetTransform()
            local parentRot = parentTransform:GetRotationWorld()
            local invParentRot = tm.quaternion.Create(-parentRot.x, -parentRot.y, -parentRot.z, parentRot.w)
            local offset = tm.vector3.op_Subtraction(position, parentTransform:GetPositionWorld())
            
            local p = tm.quaternion.Create(offset.x, offset.y, offset.z, 0)
            local p_prime = invParentRot:Multiply(p):Multiply(parentRot)
            node.localPosition = tm.vector3.Create(p_prime.x, p_prime.y, p_prime.z)
        else
            node:GetTransform():SetPositionWorld(position)
        end
        node.gridNeedsUpdate = true
    end

    local function _setNodeRotation(node, rotation)
        if node.parent then
            local parentTransform = node.parent:GetTransform()
            local parentRot = parentTransform:GetRotationWorld()
            local invParentRot = tm.quaternion.Create(-parentRot.x, -parentRot.y, -parentRot.z, parentRot.w)
            node.localRotation = invParentRot:Multiply(rotation)
        else
            node:GetTransform():SetRotationWorld(rotation)
        end
    end

    -- =================================================================
    -- OBJECT & GROUP HANDLE AND METATABLES
    -- =================================================================

    local ObjectMT = {}
    ObjectMT.__index = ObjectMT

    ObjectMT.setPosition = _setNodePosition
    ObjectMT.setRotation = _setNodeRotation

    function ObjectMT:setScale(scale)
        if not self:Exists() then return end
        local s = (type(scale) == "number") and tm.vector3.Create(scale, scale, scale) or scale
        self:GetTransform():SetScale(s)
    end

    function ObjectMT:setIsVisible(isVisible) if self:Exists() then self.gameObject:SetIsVisible(isVisible) end end
    function ObjectMT:setIsStatic(isStatic) if self:Exists() then self.gameObject:SetIsStatic(isStatic) end end
    function ObjectMT:setIsTrigger(isTrigger) if self:Exists() then self.gameObject:SetIsTrigger(isTrigger) end end
    function ObjectMT:setTexture(textureName) if self:Exists() then self.gameObject:SetTexture(textureName) end end

    function ObjectMT:despawn()
        if self.poolId then
            SceneLib.returnToPool({ handle = self })
        else
            SceneLib.despawnNode(self)
        end
    end

    function ObjectMT:GetTransform() return self.transform end

    function ObjectMT:Exists() return self.gameObject and self.gameObject:Exists() end
    
    function ObjectMT:getVelocity() 
        if not self:Exists() then return tm.vector3.Create(0,0,0) end
        return self.gameObject:GetVelocity()
    end
    
    function ObjectMT:getAngularVelocity()
        if not self:Exists() then return tm.vector3.Create(0,0,0) end
        return self.gameObject:GetAngularVelocity()
    end

    function ObjectMT:getMass()
        if not self:Exists() then return 0 end
        return self.gameObject:GetMass()
    end

    function ObjectMT:applyTorque(arg1, arg2, arg3)
        if not self:Exists() then return end
        local x, y, z
        if type(arg1) == "table" then x,y,z = arg1.x, arg1.y, arg1.z
        else x,y,z = arg1, arg2, arg3 end
        
        pcall(function()
            if self.gameObject then
               local blocks = self.gameObject.GetBlocks and self.gameObject:GetBlocks()
               if blocks and blocks[1] then
                   blocks[1]:AddTorque(x, y, z)
               end
            end
        end)
    end
    
    function ObjectMT:onClick(callback)
        if _G.InputLib then
            _G.InputLib.onObjectClicked(function(playerId, clickedObject)
                 if clickedObject and clickedObject.id == self.id then
                     callback(playerId)
                 end
            end)
        end
    end

    function ObjectMT:after(duration, callback)
        if _G.TimerLib then
            _G.TimerLib.after({
                duration = duration,
                onComplete = function()
                    if self:Exists() then callback() end
                end
            })
        end
    end
    
    function ObjectMT:raycast(direction, maxDistance, options)
        if not self:Exists() then return { hit = false } end
        options = options or {}
        options.origin = self:GetTransform():GetPositionWorld()
        options.direction = direction
        options.maxDistance = maxDistance
        options.ignoreList = options.ignoreList or { self }
        if _G.SceneLib then
            return _G.SceneLib.raycast(options)
        end
        return { hit = false }
    end

    -- =================================================================
    -- DX: WRAPPER METHODS (Convenience API)
    -- =================================================================

    function ObjectMT:animate(options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        if options.type and AnimLib[options.type] then
            AnimLib[options.type](options)
        end
    end

    function ObjectMT:move(position, duration, options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        options.targetPosition = position
        options.duration = duration
        AnimLib.tween(options)
    end

    function ObjectMT:rotate(rotation, duration, options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        options.targetRotation = rotation
        options.duration = duration
        AnimLib.tween(options)
    end
    
    function ObjectMT:scale(scale, duration, options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        options.targetScale = scale
        options.duration = duration
        AnimLib.tween(options)
    end

    function ObjectMT:spin(arg1, arg2, arg3)
        if not self:Exists() then return end
        local options = {}
        if type(arg1) == "table" and not arg1.x then
            options = arg1
        else
            options = arg3 or {}
            options.axis = arg1
            options.speed = arg2
        end
        options.handle = self
        AnimLib.spin(options)
    end

    function ObjectMT:stopAllSounds()
         if self:Exists() then
             pcall(tm.audio.StopAllAudioAtGameobject, self.gameObject)
         end
    end
    
    function ObjectMT:pulse(options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        AnimLib.pulse(options)
    end
    
    function ObjectMT:orbit(options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        AnimLib.orbit(options)
    end

    function ObjectMT:stopAnimation()
        AnimLib.stop({ handle = self })
    end
    
    function ObjectMT:follow(target, options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        options.target = target
        AnimLib.follow(options)
    end
    
    function ObjectMT:lookAt(target, options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        options.target = target
        AnimLib.lookAt(options)
    end

    function ObjectMT:applyForce(x, y, z)
        if self:Exists() then
            -- Support both vector3 and x,y,z arguments
            if type(x) == "table" and x.x and x.y and x.z then
                self.gameObject:AddForce(x.x, x.y, x.z)
            else
                self.gameObject:AddForce(x, y, z)
            end
        end
    end
    
    function ObjectMT:despawn()
        SceneLib.despawnNode(self)
    end

    function ObjectMT:bob(options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        AnimLib.bob(options)
    end
    
    function ObjectMT:sequence(options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        AnimLib.sequence(options)
    end

    function ObjectMT:path(options)
        if not self:Exists() then return end
        options = options or {}
        options.handle = self
        AnimLib.path(options)
    end

    function ObjectMT:playSound(soundName)
        if self:Exists() and soundName then
            SoundLib.playAtPosition({ sound = soundName, position = self:GetTransform():GetPositionWorld() })
        end
    end

    function ObjectMT:attachSound(soundName, id)
        if self:Exists() and soundName and id then
            SoundLib.attach({ handle = self, sound = soundName, id = id })
        end
    end
    
    function ObjectMT:stopSound(id)
        if id then
            SoundLib.stop({ id = id })
        else
            SoundLib.stopAllOnObject({ handle = self })
        end
    end

    local GroupMT = {}
    GroupMT.__index = GroupMT

    GroupMT.setPosition = _setNodePosition
    GroupMT.setRotation = _setNodeRotation

    function GroupMT:getObjects() return self.children end
    function GroupMT:despawn() SceneLib.despawnNode(self) end
    function GroupMT:Exists() return self.rootObject and self.rootObject:Exists() end
    function GroupMT:GetTransform() return self.rootTransform end

    function GroupMT:setIsVisible(isVisible)
        for _, child in ipairs(self.children) do if child:Exists() then child:setIsVisible(isVisible) end end
    end
    function GroupMT:setIsStatic(isStatic)
        for _, child in ipairs(self.children) do if child:Exists() then child:setIsStatic(isStatic) end end
    end
    function GroupMT:setIsTrigger(isTrigger)
        for _, child in ipairs(self.children) do if child:Exists() then child:setIsTrigger(isTrigger) end end
    end

    -- =================================================================
    -- DX: WRAPPER METHODS (Group API)
    -- =================================================================

    function GroupMT:animate(options)
        options = options or {}
        options.handle = self
        if options.type and AnimLib[options.type] then
            AnimLib[options.type](options)
        end
    end

    function GroupMT:move(position, duration, options)
        options = options or {}
        options.handle = self
        options.targetPosition = position
        options.duration = duration
        AnimLib.tween(options)
    end

    function GroupMT:rotate(rotation, duration, options)
        options = options or {}
        options.handle = self
        options.targetRotation = rotation
        options.duration = duration
        AnimLib.tween(options)
    end

    function GroupMT:scale(scale, duration, options)
        options = options or {}
        options.handle = self
        options.targetScale = scale
        options.duration = duration
        AnimLib.tween(options)
    end
    
    function GroupMT:spin(arg1, arg2, arg3)
        local options = {}
        if type(arg1) == "table" and not arg1.x then
            options = arg1
        else
            options = arg3 or {}
            options.axis = arg1
            options.speed = arg2
        end
        options.handle = self
        AnimLib.spin(options)
    end

    function GroupMT:stopAnimation()
        AnimLib.stop({ handle = self })
    end

    function GroupMT:bob(options)
        options = options or {}
        options.handle = self
        AnimLib.bob(options)
    end
    
    function GroupMT:sequence(options)
        options = options or {}
        options.groupHandle = self
        AnimLib.sequenceGroup(options)
    end
    
    function GroupMT:playSound(soundName)
        if soundName and self.rootObject then
            SoundLib.playAtPosition({ sound = soundName, position = self.rootTransform:GetPositionWorld() })
        end
    end

    function GroupMT:attachSound(soundName, id)
        if soundName and id then
            SoundLib.attach({ handle = self, sound = soundName, id = id })
        end
    end

    function GroupMT:stopSound(id)
        if id then
            SoundLib.stop({ id = id })
        else
            SoundLib.stopAllOnObject({ handle = self })
        end
    end

    -- =================================================================
    -- SPATIAL GRID
    -- =================================================================

    local function _getGridCoords(position)
        local x = math.floor(position.x * Grid.inverseCellSize)
        local z = math.floor(position.z * Grid.inverseCellSize)
        return x, z
    end

    local function _getGridKey(x, z)
        return x .. ":" .. z
    end

    _updateEntityInGrid = function(entity)
        if not entity or not entity:Exists() then return end
        
        local pos = entity:GetTransform():GetPositionWorld()
        local cx, cz = _getGridCoords(pos)
        local newKey = _getGridKey(cx, cz)

        if entity.gridKey ~= newKey then
            if entity.gridKey and Grid.cells[entity.gridKey] then
                Grid.cells[entity.gridKey][entity.id] = nil
            end

            if not Grid.cells[newKey] then
                Grid.cells[newKey] = {}
            end
            Grid.cells[newKey][entity.id] = entity
            entity.gridKey = newKey
        end
        entity.gridNeedsUpdate = false
    end

    local function _removeEntityFromGrid(entity)
        if entity and entity.gridKey and Grid.cells[entity.gridKey] then
            Grid.cells[entity.gridKey][entity.id] = nil
            entity.gridKey = nil
        end
    end

    -- =================================================================
    -- CREATION & SPAWNING
    -- =================================================================

    local function createObjectHandle(id, gameObject, name)
        if not gameObject then return nil end
        local handle = {
            id = id,
            type = "object",
            gameObject = gameObject,
            name = name,
            transform = gameObject:GetTransform(),
            parent = nil,
            children = {},
            tags = {},
            localPosition = tm.vector3.Create(0,0,0),
            localRotation = tm.quaternion.Create(0,0,0,1),
            poolId = nil,
            gridKey = nil,
            gridNeedsUpdate = true
        }
        setmetatable(handle, ObjectMT)
        SceneLib.trackedObjects[id] = handle
        _updateEntityInGrid(handle)
        
        -- Fire spawn event
        fireEvent("onObjectSpawned", handle)

        return handle
    end

    function SceneLib.createGroup(options)
        options = options or {}
        local id = options.id or generateId("group")
        local position = options.position or tm.vector3.Create(0,0,0)
        local rotation = options.rotation or tm.quaternion.Create(0,0,0,1)

        local rootObject = tm.physics.SpawnBoxTrigger(position, tm.vector3.Create(0.1, 0.1, 0.1))
        if not rootObject then return nil end
        rootObject:SetIsVisible(false)
        rootObject:GetTransform():SetRotationWorld(rotation)

        local groupHandle = {
            id = id,
            type = "group",
            isGroup = true,
            rootObject = rootObject,
            rootTransform = rootObject:GetTransform(),
            parent = nil,
            children = {},
            tags = {},
            localPosition = tm.vector3.Create(0,0,0),
            localRotation = tm.quaternion.Create(0,0,0,1),
            gridKey = nil,
            gridNeedsUpdate = true
        }
        setmetatable(groupHandle, GroupMT)

        if options.group and (options.group.isGroup or options.group.gameObject) then
            groupHandle.parent = options.group
            table.insert(options.group.children, groupHandle)
            groupHandle:setPosition(position)
            groupHandle:setRotation(rotation)
        end

        if options.tags and type(options.tags) == "table" then
            for _, tag in ipairs(options.tags) do groupHandle.tags[tag] = true end
        end

        SceneLib.trackedGroups[id] = groupHandle
        _updateEntityInGrid(groupHandle)

        -- Fire spawn event for the group handle
        fireEvent("onObjectSpawned", groupHandle)

        return groupHandle
    end



    -- Standard Icons for UI/Textures
    SceneLib.standardIcons = {
        "alert", "bug", "chat", "controller", "download", "edit", "error", "folder", "home",
        "hidden", "info", "lock", "menu", "plugin", "save", "search", "server", "cog",
        "success", "terminal", "trash", "unlock", "user", "visible", "warning", "wrench"
    }

    function SceneLib.preloadStandardIcons()
       local count = 0
       for _, icon in ipairs(SceneLib.standardIcons) do
           -- We register them so they can be used just by name "alert" instead of full path
           local path = "data_static/icons/" .. icon .. ".png"
           pcall(tm.physics.AddTexture, path, icon)
           count = count + 1
       end
       Logger.debug("SceneLib: Preloaded " .. count .. " standard icons.")
    end
    
    -- Auto-preload on load
    SceneLib.preloadStandardIcons()

    local function applySpawnOptions(handle, options)
        if not handle or not options then return end
        
        if options.scale then handle:setScale(options.scale) end
        if options.isStatic ~= nil then handle:setIsStatic(options.isStatic) end
        if options.isTrigger ~= nil then handle:setIsTrigger(options.isTrigger) end
        if options.isVisible ~= nil then handle:setIsVisible(options.isVisible) end

        if options.tags and type(options.tags) == "table" then
            for _, tag in ipairs(options.tags) do handle.tags[tag] = true end
        end

        if options.group and (options.group.isGroup or options.group.gameObject) then
            handle.parent = options.group
            table.insert(options.group.children, handle)
            if options.position then handle:setPosition(options.position) end
            if options.rotation then handle:setRotation(options.rotation) end
        end
    end

    function SceneLib.spawnObject(options)
        if not options or not options.name or not options.position then
            Logger.error("SceneLib: spawnObject requires a 'name' and 'position' in the options table.")
            return nil
        end
        
        local id = options.id or generateId(options.name)
        local spawnPos = (options.group and options.group:GetTransform():GetPositionWorld()) or options.position
        
        local gameObject = tm.physics.SpawnObject(spawnPos, options.name)
        local handle = createObjectHandle(id, gameObject, options.name)
        if not handle then return nil end
        
        handle:GetTransform():SetPositionWorld(options.position)
        applySpawnOptions(handle, options)
        return handle
    end

    function SceneLib.spawnCustomObject(options)
        if not options or not options.meshName or options.meshName == "" or not options.textureName or options.textureName == "" then
            Logger.error("SceneLib: spawnCustomObject requires non-empty 'meshName' and 'textureName'.")
            return nil
        end

        -- Pre-load assets as required by core API
        pcall(tm.physics.AddMesh, options.meshName .. ".obj", options.meshName)
        pcall(tm.physics.AddTexture, options.textureName .. ".png", options.textureName)

        local id = options.id or generateId(options.meshName)
        local position = options.position or tm.vector3.Create(0, 0, 0)
        local spawnPos = (options.group and options.group:GetTransform():GetPositionWorld()) or position

        local spawnFunctions = {
            RIGIDBODY = function() return tm.physics.SpawnCustomObjectRigidbody(spawnPos, options.meshName, options.textureName, options.isKinematic or false, options.mass or 1) end,
            CONCAVE = function() return tm.physics.SpawnCustomObjectConcave(spawnPos, options.meshName, options.textureName) end,
            STANDARD = function() return tm.physics.SpawnCustomObject(spawnPos, options.meshName, options.textureName) end
        }

        local objType = options.type or "STANDARD"
        local spawnFunc = spawnFunctions[objType] or spawnFunctions["STANDARD"]
        local gameObject = spawnFunc()

        if not gameObject then return nil end

        local handle = createObjectHandle(id, gameObject, options.meshName)
        handle:GetTransform():SetPositionWorld(position)
        applySpawnOptions(handle, options)
        return handle
    end

    -- =================================================================
    -- HIERARCHY & NODE MANAGEMENT
    -- =================================================================

    function SceneLib.despawnNode(node)
        if not node then return end
        
        -- Fire despawn event before any cleanup
        fireEvent("onObjectDespawned", node)
        
        _removeEntityFromGrid(node)

        for i = #node.children, 1, -1 do
            SceneLib.despawnNode(node.children[i])
        end

        if node:Exists() then
            AnimLib.stop({ handle = node })
            SoundLib.stopAllOnObject({ handle = node })
            if node.isGroup then
                node.rootObject:Despawn()
            else
                node.gameObject:Despawn()
            end
        end

        if node.isGroup then
            SceneLib.trackedGroups[node.id] = nil
        else
            SceneLib.trackedObjects[node.id] = nil
        end
    end

    function SceneLib.getObject(id)
        return SceneLib.trackedObjects[id] or SceneLib.trackedGroups[id] or SceneLib.trackedPlayers[id]
    end

    function SceneLib.cleanup()
        for id, node in pairs(SceneLib.trackedObjects) do if not node.parent then SceneLib.despawnNode(node) end end
        for id, node in pairs(SceneLib.trackedGroups) do if not node.parent then SceneLib.despawnNode(node) end end
        for id, _ in pairs(SceneLib.trackedTriggers) do SceneLib.removeTrigger({ id = id }) end
        for id, _ in pairs(SceneLib.virtualTriggers) do SceneLib.removeTrigger({ id = id }) end
        for id, poolData in pairs(SceneLib.pools) do
            for _, handle in ipairs(poolData.handles) do
                if handle and handle:Exists() then handle.gameObject:Despawn() end
            end
        end
        SceneLib.pools = {}
        SceneLib.trackedPlayers = {}
        Grid.cells = {}
        Logger.info("SceneLib: Cleaned up all tracked objects, groups, triggers, and pools.")
    end

    function SceneLib.getObjectsInRadius(position, radius, typeFilter)
        return SceneLib.findEntitiesInRadius(position, radius, { type = typeFilter })
    end

    -- =================================================================
    -- RECURSIVE UPDATE LOOP
    -- =================================================================

    updateNodeRecursively = function(node, parentNode)
        local parentTransform = parentNode:GetTransform()
        local newWorldPos = parentTransform:TransformPoint(node.localPosition)
        local newWorldRot = parentTransform:GetRotationWorld():Multiply(node.localRotation)
        
        local nodeTransform = node:GetTransform()
        nodeTransform:SetPositionWorld(newWorldPos)
        nodeTransform:SetRotationWorld(newWorldRot)
        
        node.gridNeedsUpdate = true

        for _, child in ipairs(node.children) do
            if child:Exists() then
                updateNodeRecursively(child, node)
            end
        end
    end

    function SceneLib.update()
        if PlayerLib then
            local currentPlayers = PlayerLib.getAllPlayers()
            for pid, _ in pairs(SceneLib.trackedPlayers) do
                if not currentPlayers[pid] then
                    _removeEntityFromGrid(SceneLib.trackedPlayers[pid])
                    SceneLib.trackedPlayers[pid] = nil
                end
            end
            for pid, playerObject in pairs(currentPlayers) do
                local handle = SceneLib.trackedPlayers[pid]
                if not handle then
                    handle = {
                        id = pid,
                        type = "player",
                        Exists = function() return SceneLib.trackedPlayers[pid] and SceneLib.trackedPlayers[pid].playerObject:isAlive() end,
                        GetTransform = function() return tm.players.GetPlayerTransform(pid) end,
                        gridKey = nil
                    }
                    SceneLib.trackedPlayers[pid] = handle
                end
                handle.playerObject = playerObject
                handle.name = playerObject.name
                handle.gridNeedsUpdate = true
            end
        end

        local function processRootNode(node)
            if node:Exists() and #node.children > 0 then
                for _, child in ipairs(node.children) do
                    if child:Exists() then
                        updateNodeRecursively(child, node)
                    end
                end
            end
        end
        for _, node in pairs(SceneLib.trackedObjects) do if not node.parent then processRootNode(node) end end
        for _, node in pairs(SceneLib.trackedGroups) do if not node.parent then processRootNode(node) end end

        for _, node in pairs(SceneLib.trackedObjects) do if node.gridNeedsUpdate then _updateEntityInGrid(node) end end
        for _, node in pairs(SceneLib.trackedGroups) do if node.gridNeedsUpdate then _updateEntityInGrid(node) end end
        for _, node in pairs(SceneLib.trackedPlayers) do if node.gridNeedsUpdate then _updateEntityInGrid(node) end end

        for id, trigger in pairs(SceneLib.virtualTriggers) do
            local radiusSq = trigger.radius * trigger.radius
            for _, targetHandle in ipairs(trigger.targets) do
                if targetHandle and targetHandle:Exists() then
                    local targetPos = targetHandle:GetTransform():GetPositionWorld()
                    local triggerPos = trigger.position
                    local dx, dy, dz = targetPos.x - triggerPos.x, targetPos.y - triggerPos.y, targetPos.z - triggerPos.z
                    local distSq = dx*dx + dy*dy + dz*dz
                    local targetId = targetHandle.id
                    local isInside = distSq <= radiusSq
                    local wasInside = trigger.playersInside[targetId]

                    if isInside and not wasInside then
                        if not (trigger.oneTime and trigger.triggeredPlayers[targetId]) then
                            if type(trigger.onEnter) == "function" then pcall(trigger.onEnter, targetId, trigger) end
                            trigger.triggeredPlayers[targetId] = true
                        end
                    elseif not isInside and wasInside then
                        if type(trigger.onExit) == "function" then pcall(trigger.onExit, targetId, trigger) end
                    end
                    trigger.playersInside[targetId] = isInside
                end
            end
        end
    end

    -- =================================================================
    -- TAGGING & QUERIES
    -- =================================================================

    function SceneLib.addTag(handle, tag) if handle and tag and handle.tags then handle.tags[tag] = true end end
    function SceneLib.removeTag(handle, tag) if handle and tag and handle.tags then handle.tags[tag] = nil end end
    function SceneLib.hasTag(handle, tag)
        if not handle or not handle.tags then return false end
        return handle.tags[tag] == true
    end

    local function _passesAllFilters(entity, options)
        if options.type and entity.type ~= options.type then return false end
        if options.name and entity.name ~= options.name then return false end
        
        if options.tag then
            if not entity.tags then return false end
            local requiredTags = type(options.tag) == "table" and options.tag or {options.tag}
            for _, requiredTag in ipairs(requiredTags) do
                if not entity.tags[requiredTag] then
                    return false
                end
            end
        end
        
        return true
    end


    local function findAllEntitiesByCondition(baseConditionFunc, options)
        options = options or {}
        local results = {}
        local checkedIds = {}

        local function processCell(cell)
            for id, entity in pairs(cell) do
                if not checkedIds[id] then
                    if _passesAllFilters(entity, options) and baseConditionFunc(entity) then
                        table.insert(results, entity)
                    end
                    checkedIds[id] = true
                end
            end
        end

        if options._searchCells then
            for _, key in ipairs(options._searchCells) do
                if Grid.cells[key] then processCell(Grid.cells[key]) end
            end
        else
            for _, cell in pairs(Grid.cells) do processCell(cell) end
        end
        
        return results
    end

    function SceneLib.findEntitiesByTag(tag, options)
        if not tag then return {} end
        options = options or {}
        options.tag = tag
        return findAllEntitiesByCondition(function() return true end, options)
    end

    function SceneLib.findEntitiesByName(name, options)
        if not name then return {} end
        options = options or {}
        options.name = name
        return findAllEntitiesByCondition(function() return true end, options)
    end

    function SceneLib.findEntitiesInRadius(position, radius, options)
        options = options or {}
        local radiusSq = radius * radius
        local px, py, pz = position.x, position.y, position.z
        
        -- Optimized grid bounds calculation
        local minX, minZ = _getGridCoords({x = px - radius, y = 0, z = pz - radius}) -- Duck-typing vector
        local maxX, maxZ = _getGridCoords({x = px + radius, y = 0, z = pz + radius})
        
        local cellsToSearch = {}
        for x = minX, maxX do
            for z = minZ, maxZ do
                table.insert(cellsToSearch, _getGridKey(x, z))
            end
        end
        options._searchCells = cellsToSearch

        return findAllEntitiesByCondition(function(node)
            if not node:Exists() then return false end
            local np = node:GetTransform():GetPositionWorld() 
            local dx, dy, dz = np.x - px, np.y - py, np.z - pz
            return (dx*dx + dy*dy + dz*dz) <= radiusSq
        end, options)
    end

    function SceneLib.findEntitiesInBox(center, size, options)
        options = options or {}
        local cx, cy, cz = center.x, center.y, center.z
        local sx, sy, sz = size.x * 0.5, size.y * 0.5, size.z * 0.5
        
        local minXVal, minYVal, minZVal = cx - sx, cy - sy, cz - sz
        local maxXVal, maxYVal, maxZVal = cx + sx, cy + sy, cz + sz
        
        local gridMinX, gridMinZ = _getGridCoords({x = minXVal, z = minZVal})
        local gridMaxX, gridMaxZ = _getGridCoords({x = maxXVal, z = maxZVal})
        
        local cellsToSearch = {}
        for x = gridMinX, gridMaxX do
            for z = gridMinZ, gridMaxZ do table.insert(cellsToSearch, _getGridKey(x, z)) end
        end
        options._searchCells = cellsToSearch

        return findAllEntitiesByCondition(function(node)
            if not node:Exists() then return false end
            local pos = node:GetTransform():GetPositionWorld()
            return pos.x >= minXVal and pos.x <= maxXVal and pos.y >= minYVal and pos.y <= maxYVal and pos.z >= minZVal and pos.z <= maxZVal
        end, options)
    end

    function SceneLib.findClosestEntity(position, options)
        options = options or {}
        local searchRadius = options.maxDistance or Grid.cellSize
        local entitiesToSearch = SceneLib.findEntitiesInRadius(position, searchRadius, options)
        local closestEntity, minDistSq = nil, searchRadius * searchRadius
        
        for _, entity in ipairs(entitiesToSearch) do
            if entity:Exists() then
                local ePos = entity:GetTransform():GetPositionWorld()
                local dx, dy, dz = ePos.x - position.x, ePos.y - position.y, ePos.z - position.z
                local distSq = dx*dx + dy*dy + dz*dz
                
                if distSq < minDistSq then
                    minDistSq, closestEntity = distSq, entity
                end
            end
        end
        
        return closestEntity
    end

    -- =================================================================
    -- OBJECT POOLING
    -- =================================================================

    function SceneLib.createPool(options)
        if not options or not options.id or not options.objectName or not options.size then
            Logger.error("SceneLib: createPool requires id, objectName, and size.")
            return
        end
        
        SceneLib.pools[options.id] = { handles = {}, config = { recycleWhenExhausted = options.recycleWhenExhausted ~= false } }

        for i = 1, options.size do
            local handle = SceneLib.spawnObject({ name = options.objectName, position = tm.vector3.Create(0,-10000,0) })
            if handle then
                handle:setIsVisible(false); handle:setIsStatic(true)
                handle.isAvailable = true; handle.poolId = options.id
                table.insert(SceneLib.pools[options.id].handles, handle)
            end
        end
        Logger.info("SceneLib: Created object pool '" .. options.id .. "' with " .. options.size .. " objects.")
    end

    function SceneLib.getFromPool(options)
        if not options or not options.id then return nil end
        local poolData = SceneLib.pools[options.id]
        if not poolData then return nil end

        local handles, config = poolData.handles, poolData.config
        local objectToUse, foundAtIndex = nil

        for i, handle in ipairs(handles) do
            if handle.isAvailable then objectToUse, foundAtIndex = handle, i; break end
        end

        if not objectToUse then
            if config.recycleWhenExhausted then
                objectToUse, foundAtIndex = handles[1], 1
                AnimLib.stop({ handle = objectToUse }); objectToUse:setIsVisible(false)
            else return nil end
        end
        
        table.remove(handles, foundAtIndex); table.insert(handles, objectToUse)
        objectToUse.isAvailable = false; objectToUse:setIsVisible(false)

        local spawnOptions = options.spawnOptions or {}
        if spawnOptions.position then objectToUse:setPosition(spawnOptions.position) end
        if spawnOptions.rotation then objectToUse:setRotation(spawnOptions.rotation) end
        objectToUse:setIsStatic(spawnOptions.isStatic == true)
        
        if spawnOptions.isVisible ~= false then
            TimerLib.after({ duration = 0.05, onComplete = function() if objectToUse and objectToUse:Exists() then objectToUse:setIsVisible(true) end end })
        end
        
        return objectToUse
    end

    function SceneLib.returnToPool(options)
        local handle = options.handle
        if not handle or not handle.poolId or not SceneLib.pools[handle.poolId] then return end
        
        handle:setIsVisible(false); handle:setIsStatic(true)
        handle:setPosition(tm.vector3.Create(0,-10000,0))
        AnimLib.stop({ handle = handle }); handle.isAvailable = true
    end

    -- =================================================================
    -- TRIGGER MANAGEMENT
    -- =================================================================

    local function registerTriggerCallbacks(trigger)
        if not trigger.triggerBox then return end
        
        local enterFuncName = "SceneLib_TriggerEnter_" .. trigger.id
        local exitFuncName = "SceneLib_TriggerExit_" .. trigger.id
        
        _G[enterFuncName] = function(playerId)
            local currentTrigger = SceneLib.trackedTriggers[trigger.id]
            if not currentTrigger or (currentTrigger.oneTime and currentTrigger.triggeredPlayers[playerId]) then return end
            
            local callback = currentTrigger.onEnter
            if type(callback) == "function" then pcall(callback, playerId, currentTrigger) end
            currentTrigger.triggeredPlayers[playerId] = true
        end

        _G[exitFuncName] = function(playerId)
            local currentTrigger = SceneLib.trackedTriggers[trigger.id]
            if not currentTrigger then return end
            if type(currentTrigger.onExit) == "function" then pcall(currentTrigger.onExit, playerId, currentTrigger) end
        end

        tm.physics.RegisterFunctionToCollisionEnterCallback(trigger.triggerBox, enterFuncName)
        tm.physics.RegisterFunctionToCollisionExitCallback(trigger.triggerBox, exitFuncName)
    end

    function SceneLib.createTrigger(options)
        options = options or {}
        local id = options.id or generateId("trigger")
        local triggerType = options.type or "AREA"

        if triggerType == "VIRTUAL" then
            if not options.targets or not options.position or not options.radius then
                Logger.error("SceneLib: Virtual trigger requires 'targets', 'position', and 'radius'.")
                return nil
            end
            local virtualTrigger = {
                id = id,
                type = "VIRTUAL",
                onEnter = options.onEnter,
                onExit = options.onExit,
                oneTime = options.oneTime or false,
                triggeredPlayers = {},
                playersInside = {},
                position = options.position,
                radius = options.radius,
                targets = options.targets,
                data = options.data or {}
            }
            SceneLib.virtualTriggers[id] = virtualTrigger
            
            if options.duration and options.duration > 0 then
                TimerLib.after({
                    duration = options.duration,
                    onComplete = function() SceneLib.removeTrigger({ id = id }) end
                })
            end
            
            return virtualTrigger
        end

        local trackedTrigger = {
            id = id,
            type = triggerType,
            onEnter = options.onEnter,
            onExit = options.onExit,
            oneTime = options.oneTime or false,
            triggeredPlayers = {},
            position = options.position or tm.vector3.Create(0, 0, 0),
            size = options.size or tm.vector3.Create(5, 5, 5),
            data = options.data or {}
        }

        local triggerGameObject = tm.physics.SpawnBoxTrigger(trackedTrigger.position, trackedTrigger.size)
        if not triggerGameObject then return nil end
        triggerGameObject:SetIsVisible(options.isVisible == true)

        local triggerHandle = createObjectHandle("trigger_handle_" .. id, triggerGameObject, "TriggerBox")
        if not triggerHandle then
            triggerGameObject:Despawn()
            return nil
        end
        
        trackedTrigger.triggerHandle = triggerHandle
        trackedTrigger.triggerBox = triggerGameObject

        if triggerType == "ATTACHED" then
            if not options.target then
                Logger.error("SceneLib: Attached trigger requires a 'target' to follow.")
                SceneLib.despawnNode(triggerHandle)
                return nil
            end
            
            AnimLib.follow({
                handle = triggerHandle,
                target = options.target,
                positionOffset = options.offset or tm.vector3.Create(0,0,0),
                smoothing = options.smoothing or 0
            })
        end
        
        registerTriggerCallbacks(trackedTrigger)
        SceneLib.trackedTriggers[id] = trackedTrigger
        
        if options.duration and options.duration > 0 then
            TimerLib.after({
                duration = options.duration,
                onComplete = function() SceneLib.removeTrigger({ id = id }) end
            })
        end
        
        return trackedTrigger
    end

    function SceneLib.removeTrigger(options)
        if not options or not options.id then return end
        local id = options.id
        local physicalTrigger = SceneLib.trackedTriggers[id]
        local virtualTrigger = SceneLib.virtualTriggers[id]

        if physicalTrigger then
            if physicalTrigger.triggerHandle and physicalTrigger.triggerHandle:Exists() then
                SceneLib.despawnNode(physicalTrigger.triggerHandle)
            end
            _G["SceneLib_TriggerEnter_" .. id] = nil
            _G["SceneLib_TriggerExit_" .. id] = nil
            SceneLib.trackedTriggers[id] = nil
        elseif virtualTrigger then
            SceneLib.virtualTriggers[id] = nil
        end
    end

    -- =================================================================
    -- RAYCASTING
    -- =================================================================

    local function findTargetAtPoint(hitPoint)
        for id, playerHandle in pairs(SceneLib.trackedPlayers) do
            if playerHandle:Exists() then
                local playerPos = playerHandle:GetTransform():GetPositionWorld()
                if tm.vector3.Distance(hitPoint, playerPos) < 1.5 then
                    return playerHandle, "player"
                end
            end
        end

        local closestObject = SceneLib.findClosestEntity(hitPoint, { maxDistance = 5 })
        if closestObject and closestObject.type ~= "player" then
            return closestObject, "spawnedObject"
        end

        return nil, "terrain"
    end

    function SceneLib.raycast(options)
        options = options or {}
        local origin = options.origin
        local direction = options.direction
        if not origin or not direction then return { hit = false } end

        local maxDistance = options.maxDistance or 1000
        local ignoreList = options.ignoreList or {}
        local currentOrigin, remainingDistance, totalDistanceTraveled, iterations = origin, maxDistance, 0, 0

        local function isIgnored(target)
            if not target or #ignoreList == 0 then return false end
            for _, ignoredHandle in ipairs(ignoreList) do if target.id == ignoredHandle.id then return true end end
            return false
        end

        local function isTypeAllowed(type)
            if not options.filter then return true end
            if type(options.filter) == "string" then return options.filter == type end
            if type(options.filter) == "table" then
                for _, f in ipairs(options.filter) do if f == type then return true end end
            end
            return false
        end

        while remainingDistance > 0.01 and iterations < 10 do
            iterations = iterations + 1
            local rawHit = tm.physics.RaycastData(currentOrigin, direction, remainingDistance, options.ignoreTriggers or false)
            if not rawHit or not rawHit:DidHit() then return { hit = false } end

            local hitPoint, hitDistance = rawHit:GetHitPosition(), rawHit:GetHitDistance()
            totalDistanceTraveled = totalDistanceTraveled + hitDistance
            local target, hitType = findTargetAtPoint(hitPoint)

            if not isIgnored(target) and isTypeAllowed(hitType) then
                return { hit = true, point = hitPoint, normal = rawHit:GetHitNormal(), distance = totalDistanceTraveled, type = hitType, target = target }
            else
                currentOrigin = tm.vector3.op_Addition(hitPoint, tm.vector3.op_Multiply(direction, 0.01))
                remainingDistance = remainingDistance - hitDistance - 0.01
            end
        end
        return { hit = false }
    end

    -- =================================================================
    -- PUBLIC EVENT REGISTRATION
    -- =================================================================

    -- Functions to register callbacks for the new events
    function SceneLib.onObjectSpawned(callback)
        if type(callback) == "function" then
            table.insert(SceneLib.events.onObjectSpawned, callback)
        end
    end

    function SceneLib.onObjectDespawned(callback)
        if type(callback) == "function" then
            table.insert(SceneLib.events.onObjectDespawned, callback)
        end
    end


    -- =================================================================
    -- FINAL REGISTRATION
    -- =================================================================

    UpdateManager.register(SceneLib.update, 900)

    return SceneLib
end)()

-- =================================================================
-- MODULE: PHYSICS LIB
-- =================================================================
_G.PhysicsLib = (function()
    local PhysicsLib = {}
    -- tm.os.SetModTargetDeltaTime(1/60) -- Global setting

    local UpdateManager = _G.UpdateManager
    
    local TimerLib = _G.TimerLib
    local SceneLib = _G.SceneLib

    local LOG_PREFIX = "PhysicsLib: "
    
    -- =================================================================
    -- CACHED MATH FUNCTIONS
    -- =================================================================
    local sqrt, sin, cos, acos, asin, deg, rad = math.sqrt, math.sin, math.cos, math.acos, math.asin, math.deg, math.rad
    local min, max, abs, floor = math.min, math.max, math.abs, math.floor
    local v3Create = tm.vector3.Create -- Cache constructor
    
    -- =================================================================
    -- CONFIGURATION & STATE
    -- =================================================================
    
    local structuresWithConstantForce, structuresFollowingPath = {}, {}
    
    local CONFIG = {
        -- Path following configuration
        LOOK_AHEAD_POINTS = 12, 
        MIN_SPEED_MULTIPLIER = 0.7, 
        MAX_SPEED_MULTIPLIER = 1,
        SHARP_TURN_THRESHOLD = 25, 
        STUCK_TIME_THRESHOLD = 20.0, 
        MAX_PATH_DEVIATION = 300,
        MIN_MOVEMENT_THRESHOLD = 0.01, 
        RESPAWN_HEIGHT_OFFSET = 2.0, 
        RECOVERY_COOLDOWN = 3.0,
        HEALTH_CHECK_DELAY = 2.0, 
        HILL_SPEED_BOOST = 0.3, 
        HILL_SPEED_REDUCTION = 0.2,
        MIN_HILL_ANGLE = 5, 
        MAX_HILL_ANGLE = 45, 
        TURN_TORQUE_BOOST = 1.2, 
        MAX_TURN_TORQUE_MULTIPLIER = 1.5,
        PATH_CORRECTION_STRENGTH = 6000,
        
        -- Banking configuration
        MAX_BANK_ANGLE = 45,
        BANK_RESPONSIVENESS = 1.0,
        BANK_TO_YAW_RATIO = 0.5,
        ROLL_STRENGTH = 3.0,
        MIN_YAW_FOR_BANK = 0.02,
        BANK_LIFT_FACTOR = 0.1,
        PITCH_STABILITY_FACTOR = 0.5,
    }
    
    -- =================================================================
    -- OPTIMIZED UTILITIES
    -- =================================================================
    
    -- Standard distance calculation (Scalar)
    local function dist(p1, p2)
        if not p1 or not p2 then return 1000000 end
        local dx, dy, dz = p1.x - p2.x, p1.y - p2.y, p1.z - p2.z
        return sqrt(dx*dx + dy*dy + dz*dz)
    end
    
    -- Optimized normalization returning x,y,z (No allocation)
    local function normalizeXYZ(x, y, z)
        local m = sqrt(x*x + y*y + z*z)
        if m > 0 then
            local inv = 1/m
            return x*inv, y*inv, z*inv
        end
        return 0, 0, 0
    end

    -- Legacy helper returning vector (kept for compatibility if needed elsewhere, though avoiding in loops)
    local function norm(v)
        local mx, my, mz = v.x, v.y, v.z
        local m = sqrt(mx*mx + my*my + mz*mz)
        if m > 0 then
            local inv = 1/m
            return v3Create(mx*inv, my*inv, mz*inv)
        end
        return v3Create(0,0,0)
    end
    
    -- Optimized angle calculation
    local function angle(v1, v2)
        local d = v1.x*v2.x + v1.y*v2.y + v1.z*v2.z
        local m1 = sqrt(v1.x*v1.x + v1.y*v1.y + v1.z*v1.z)
        local m2 = sqrt(v2.x*v2.x + v2.y*v2.y + v2.z*v2.z)
        if m1 > 0 and m2 > 0 then
            return deg(acos(max(-1, min(1, d/(m1*m2)))))
        end
        return 0
    end
    
    -- Scalar Cross Product returning x,y,z
    local function crossXYZ(ax, ay, az, bx, by, bz)
        return ay*bz - az*by, az*bx - ax*bz, ax*by - ay*bx
    end

    local function cross(v1, v2)
        return v3Create(v1.y*v2.z - v1.z*v2.y, v1.z*v2.x - v1.x*v2.z, v1.x*v2.y - v1.y*v2.x)
    end
    
    local function dot(v1, v2)
        return v1.x * v2.x + v1.y * v2.y + v1.z * v2.z
    end
    
    local clamp = function(v, minVal, maxVal) return max(minVal, min(maxVal, v)) end
    local lerp = function(a, b, t) return a + (b-a)*t end
    
    -- Keep original findNearestPathPoint for correctness
    local function findNearestPathPoint(path, pos, startIdx, endIdx)
        local nearest, minDist = startIdx or 1, 1000000
        for i = (startIdx or 1), (endIdx or #path) do
            local d = dist(pos, path[i])
            if d < minDist then 
                minDist, nearest = d, i 
            end
        end
        return nearest
    end
    
    -- =================================================================
    -- PID CONTROLLER
    -- =================================================================
    
    local function createPIDController(p, i, d, maxI)
        return { 
            p = p or 1.0, 
            i = i or 0.0, 
            d = d or 0.0, 
            integral = 0, 
            lastError = 0,
            maxIntegral = maxI or 2
        }
    end
    
    local function updatePID(pid, error, dt)
        if dt == 0 then return 0 end
        
        -- Update integral with anti-windup
        pid.integral = pid.integral + error * dt
        pid.integral = clamp(pid.integral, -pid.maxIntegral, pid.maxIntegral)
        
        -- Calculate derivative
        local derivative = (error - pid.lastError) / dt
        pid.lastError = error
        
        -- Calculate PID output
        return (pid.p * error) + (pid.i * pid.integral) + (pid.d * derivative)
    end
    
    -- =================================================================
    -- BÉZIER CURVE PATH GENERATION (OPTIMIZED)
    -- =================================================================
    
    local function getBezierPoint(p0, p1, p2, p3, t)
        local u = 1 - t
        local tt = t*t
        local uu = u*u
        local uuu = uu*u
        local ttt = tt*t
        
        -- Pre-calculate coefficients
        local c0 = uuu
        local c1 = 3 * uu * t
        local c2 = 3 * u * tt
        local c3 = ttt
        
        -- Direct calculation instead of multiple vector operations
        return v3Create(
            p0.x * c0 + p1.x * c1 + p2.x * c2 + p3.x * c3,
            p0.y * c0 + p1.y * c1 + p2.y * c2 + p3.y * c3,
            p0.z * c0 + p1.z * c1 + p2.z * c2 + p3.z * c3
        )
    end
    
    local function generateBezierPath(waypoints, loop, numPointsPerSegment, visualize)
        local path, waypointsCount = {}, #waypoints
        if waypointsCount < 2 then return waypoints end
        
        if loop and dist(waypoints[1], waypoints[waypointsCount]) < 0.01 then 
            table.remove(waypoints)
            waypointsCount = #waypoints 
        end
        
        local controlPoints1, controlPoints2 = {}, {}
        for i = 1, waypointsCount do
            local prev_i = (i == 1) and (loop and waypointsCount or 1) or i - 1
            local next_i = (i == waypointsCount) and (loop and 1 or waypointsCount) or i + 1
            local prev_p, current_p, next_p = waypoints[prev_i], waypoints[i], waypoints[next_i]
            local tanX, tanY, tanZ = normalizeXYZ(next_p.x - prev_p.x, next_p.y - prev_p.y, next_p.z - prev_p.z)
            local dist_prev, dist_next = dist(current_p, prev_p), dist(current_p, next_p)
            
            -- c1 = current - tangent * (dist * 0.33)
            controlPoints1[i] = v3Create(current_p.x - tanX * dist_prev * 0.33, current_p.y - tanY * dist_prev * 0.33, current_p.z - tanZ * dist_prev * 0.33)
            -- c2 = current + tangent * (dist * 0.33)
            controlPoints2[i] = v3Create(current_p.x + tanX * dist_next * 0.33, current_p.y + tanY * dist_next * 0.33, current_p.z + tanZ * dist_next * 0.33)
        end
        
        for i = 1, waypointsCount do
            if not loop and i == waypointsCount then break end 
            local next_i = (i == waypointsCount) and 1 or i + 1
            local p0, p1, p2, p3 = waypoints[i], controlPoints2[i], controlPoints1[next_i], waypoints[next_i]
            for j = 0, numPointsPerSegment - 1 do
                local t = j / numPointsPerSegment
                table.insert(path, getBezierPoint(p0, p1, p2, p3, t))
            end
        end
        
        if not loop then
            table.insert(path, waypoints[waypointsCount])
        end
        
        return path
    end
    
    -- =================================================================
    -- HEALTH & RECOVERY
    -- =================================================================
    
    local function initHealthTracking() 
        return { 
            lastPosition = nil, 
            lastMoveTime = tm.os.GetTime(), 
            stuckTime = 0, 
            lastRecoveryTime = 0, 
            recoveryAttempts = 0, 
            startTime = tm.os.GetTime() 
        } 
    end
    
    local function checkHealth(pathData, currentPos) 
        local health, currentTime = pathData.health, tm.os.GetTime() 
        if currentTime - health.startTime < CONFIG.HEALTH_CHECK_DELAY then return false, "initializing" end 
        if not health.lastPosition then 
            health.lastPosition = currentPos 
            health.lastMoveTime = currentTime 
            return false, "first_position" 
        end 
        
        local movementDistance = dist(currentPos, health.lastPosition) 
        if movementDistance > CONFIG.MIN_MOVEMENT_THRESHOLD then 
            health.lastMoveTime = currentTime 
            health.stuckTime = 0 
            health.lastPosition = currentPos 
        else 
            health.stuckTime = currentTime - health.lastMoveTime 
        end 
        
        if health.stuckTime > CONFIG.STUCK_TIME_THRESHOLD and (currentTime - health.lastRecoveryTime) > CONFIG.RECOVERY_COOLDOWN then 
            return true, "stuck" 
        end 
        
        local nearestPathPoint = pathData.path[pathData.currentWaypoint] 
        if not nearestPathPoint then return false, "healthy" end 
        
        local pathDeviation = dist(currentPos, nearestPathPoint) 
        if pathDeviation > CONFIG.MAX_PATH_DEVIATION and (currentTime - health.lastRecoveryTime) > CONFIG.RECOVERY_COOLDOWN then 
            return true, "off_path" 
        end 
        
        return false, "healthy" 
    end
    
    local function advancePathProgress(data, currentPos)
        local path = data.path
        local pathSize = #path
        if pathSize == 0 then return 1 end
    
        local currentIndex = data.currentWaypoint
        local currentPointDist = dist(currentPos, path[currentIndex])
    
        for i = 1, 5 do
            local nextIndex = currentIndex + i
            if nextIndex > pathSize then
                if data.loop then
                    nextIndex = (nextIndex - 1) % pathSize + 1
                else
                    break
                end
            end
    
            local nextPointDist = dist(currentPos, path[nextIndex])
    
            if nextPointDist < currentPointDist then
                data.currentWaypoint = nextIndex
                currentPointDist = nextPointDist
            else
                break
            end
        end
        return data.currentWaypoint
    end
    
    local function recoverVehicle(structureId, pathData, reason) 
        Logger.info("Recovering vehicle " .. structureId .. " due to: " .. reason) 
        local currentPos = pathData.handle:GetPosition() 
        local nearestIndex = findNearestPathPoint(pathData.path, currentPos, 1, #pathData.path) 
        local respawnPoint = pathData.path[nearestIndex] 
        local respawnPosition = v3Create(respawnPoint.x, respawnPoint.y + CONFIG.RESPAWN_HEIGHT_OFFSET, respawnPoint.z) 
        pathData.awaitingRespawn = true 
        pathData.handle = nil 
        tm.players.DespawnStructure(structureId) 
        TimerLib.after({ 
            duration = 0.5, 
            onComplete = function() 
                tm.players.SpawnStructure(pathData.playerId, pathData.blueprintName, structureId, respawnPosition, v3Create(0, 0, 0)) 
                pathData.currentWaypoint = nearestIndex 
                pathData.health.lastRecoveryTime = tm.os.GetTime() 
                pathData.health.recoveryAttempts = pathData.health.recoveryAttempts + 1 
                pathData.health.lastPosition = nil 
                pathData.health.startTime = tm.os.GetTime() 
            end 
        }) 
    end
    
    -- =================================================================
    -- BLOCK WRAPPER
    -- =================================================================
    local BlockMT = {}
    BlockMT.__index = BlockMT

    function BlockMT:setMass(mass) pcall(self.native.SetMass, mass) end
    function BlockMT:getMass() return self.native:GetMass() end
    function BlockMT:setDrag(fwd, back, up, down, left, right) pcall(self.native.SetDragAll, fwd, back, up, down, left, right) end
    function BlockMT:setPrimaryColor(r, g, b) 
        if type(r) == "table" then
             pcall(self.native.SetPrimaryColor, r)
        else
             pcall(self.native.SetPrimaryColor, r, g, b) 
        end
    end
    function BlockMT:setSecondaryColor(r, g, b)
        if type(r) == "table" then
             pcall(self.native.SetSecondaryColor, r)
        else
             pcall(self.native.SetSecondaryColor, r, g, b) 
        end
    end
    function BlockMT:getPrimaryColor() return self.native:GetPrimaryColor() end
    function BlockMT:getSecondaryColor() return self.native:GetSecondaryColor() end
    function BlockMT:setEnginePower(power) pcall(self.native.SetEnginePower, power) end
    function BlockMT:setJetPower(power) pcall(self.native.SetJetPower, power) end
    function BlockMT:setPropellerPower(power) pcall(self.native.SetPropellerPower, power) end
    function BlockMT:setGyroPower(power) pcall(self.native.SetGyroPower, power) end
    function BlockMT:getEnginePower() return self.native:GetEnginePower() end
    function BlockMT:getJetPower() return self.native:GetJetPower() end
    function BlockMT:getPropellerPower() return self.native:GetPropellerPower() end
    function BlockMT:getGyroPower() return self.native:GetGyroPower() end
    function BlockMT:addForce(x, y, z) pcall(self.native.AddForce, x, y, z) end
    function BlockMT:addTorque(x, y, z) pcall(self.native.AddTorque, x, y, z) end
    function BlockMT:exists() return self.native:Exists() end
    function BlockMT:getStructure() 
        local s = self.native:GetStructure()
        if s and PhysicsLib.Structure then return PhysicsLib.Structure(s) end
        return s 
    end
    
    function PhysicsLib.Block(nativeBlock)
        if not nativeBlock then return nil end
        local obj = { native = nativeBlock }
        setmetatable(obj, BlockMT)
        return obj
    end

    -- =================================================================
    -- STRUCTURE WRAPPER
    -- =================================================================
    local StructureMT = {}
    StructureMT.__index = StructureMT

    function StructureMT:getPosition() return self.native:GetPosition() end
    function StructureMT:getRotation() return self.native:GetRotation() end
    function StructureMT:getScale() return self.native:GetScale() end
    function StructureMT:transformPoint(p) return self.native:TransformPoint(p) end
    function StructureMT:transformDirection(d) return self.native:TransformDirection(d) end
    function StructureMT:forward() return self.native:Forward() end
    function StructureMT:back() return self.native:Back() end
    function StructureMT:left() return self.native:Left() end
    function StructureMT:right() return self.native:Right() end
    function StructureMT:up() return self.native:Up() end
    function StructureMT:down() return self.native:Down() end
    function StructureMT:destroy() pcall(self.native.Destroy) end
    function StructureMT:getBlocks()
        local nativeBlocks = self.native:GetBlocks()
        if not nativeBlocks then return {} end
        local wrapped = {}
        for i, b in ipairs(nativeBlocks) do
            if PhysicsLib.Block then
                table.insert(wrapped, PhysicsLib.Block(b))
            else
                table.insert(wrapped, b)
            end
        end
        return wrapped
    end
    function StructureMT:addForce(x, y, z) pcall(self.native.AddForce, x, y, z) end
    function StructureMT:getVelocity() return self.native:GetVelocity() end
    function StructureMT:getSeatedPlayerVelocity() return self.native:GetSeatedPlayerVelocity() end
    function StructureMT:getSpeed() return self.native:GetSpeed() end
    function StructureMT:getOwnerId() return self.native:GetOwnedByPlayerId() end
    function StructureMT:getPowerCores() return self.native:GetPowerCores() end
    function StructureMT:getCenterOfMass() return self.native:GetWorldCenterOfMass() end

    function PhysicsLib.Structure(nativeStructure)
        if not nativeStructure then return nil end
        local obj = { native = nativeStructure }
        setmetatable(obj, StructureMT)
        return obj
    end

    -- =================================================================
    -- PHYSICS SYSTEMS (OPTIMIZED - SCALAR MATH)
    -- =================================================================
    
    local function applyAirPhysics(data, currentPos, targetPos, id)
        local handle = data.handle
        local forward, up, right, velocity
        -- Get vectors as components if possible? API returns Vector3, so we unpack
        local success = pcall(function()
            forward = handle:Forward()
            up = handle:Up()
            right = handle:Right()
            velocity = handle:GetVelocity()
        end)
        if not success then data.awaitingRespawn = true; data.handle = nil; return end
    
        -- Unpack common components for scalar math
        local fx, fy, fz = forward.x, forward.y, forward.z
        local ux, uy, uz = up.x, up.y, up.z
        local rx, ry, rz = right.x, right.y, right.z
        local cpx, cpy, cpz = currentPos.x, currentPos.y, currentPos.z
        local tpx, tpy, tpz = targetPos.x, targetPos.y, targetPos.z
        
        -- Apply constant forward thrust
        handle:AddForce(fx * data.thrustForce, fy * data.thrustForce, fz * data.thrustForce)
    
        local dt = tm.os.GetModDeltaTime()
        if dt == 0 then return end
        
        -- Path correction (Sucking back to path line)
        local nearestPathPoint = data.path[data.currentWaypoint]
        if nearestPathPoint then
            local npx, npy, npz = nearestPathPoint.x, nearestPathPoint.y, nearestPathPoint.z
            local corX, corY, corZ = npx - cpx, npy - cpy, npz - cpz
            
            -- Correction force = vector * strength * dt
            local cfFactor = CONFIG.PATH_CORRECTION_STRENGTH * dt
            handle:AddForce(corX * cfFactor, corY * cfFactor, corZ * cfFactor)
        end
    
        -- Calculate navigation errors using scalar math
        local toTargetX, toTargetY, toTargetZ = normalizeXYZ(tpx - cpx, tpy - cpy, tpz - cpz)
        local fwdHorX, fwdHorY, fwdHorZ = normalizeXYZ(fx, 0, fz)
        local toTgtHorX, toTgtHorY, toTgtHorZ = normalizeXYZ(toTargetX, 0, toTargetZ)
        
        -- Yaw error (Cross product Y component of horizontal vectors)
        -- Cross(A, B).y = A.z * B.x - A.x * B.z
        local yawError = fwdHorZ * toTgtHorX - fwdHorX * toTgtHorZ
        
        -- Pitch error with stability bias
        local velX, velY, velZ = velocity.x, velocity.y, velocity.z
        local _, velNormY, _ = normalizeXYZ(velX, velY, velZ)
        local pitchError = (toTargetY - velNormY) * CONFIG.PITCH_STABILITY_FACTOR
        
        -- Initialize banking state
        if not data.targetBankAngle then data.targetBankAngle = 0 end
        if not data.smoothedBankAngle then data.smoothedBankAngle = 0 end
        
        -- Calculate desired bank angle based on yaw turn sharpness
        local desiredBank = 0
        if abs(yawError) > CONFIG.MIN_YAW_FOR_BANK then
            desiredBank = clamp(-yawError * CONFIG.MAX_BANK_ANGLE * 3, -CONFIG.MAX_BANK_ANGLE, CONFIG.MAX_BANK_ANGLE)
        end
        
        -- Smooth the target bank angle
        data.targetBankAngle = data.targetBankAngle + (desiredBank - data.targetBankAngle) * CONFIG.BANK_RESPONSIVENESS * dt
        data.smoothedBankAngle = lerp(data.smoothedBankAngle, data.targetBankAngle, 0.5)
        
        -- Get current bank angle (arcsin of Right.y)
        local currentBank = deg(asin(clamp(ry, -1, 1)))
        
        -- Calculate roll error
        local rollError = (currentBank - data.smoothedBankAngle) / CONFIG.MAX_BANK_ANGLE
        
        -- When banking, reduce yaw input proportionally to avoid oversteering into the ground/sky
        local bankFactor = abs(data.smoothedBankAngle) / CONFIG.MAX_BANK_ANGLE
        local adjustedYawError = yawError * (1 - bankFactor * CONFIG.BANK_TO_YAW_RATIO)
        
        -- Update PID controllers
        local yawTorque = updatePID(data.yawPID, adjustedYawError, dt) * data.steeringTorque
        local pitchTorque = updatePID(data.pitchPID, -pitchError, dt) * data.steeringTorque
        local rollTorque = -rollError * data.steeringTorque * CONFIG.ROLL_STRENGTH
        
        -- Apply torques: Total = Up*Yaw + Right*Pitch + Forward*Roll
        local tx = ux * yawTorque + rx * pitchTorque + fx * rollTorque
        local ty = uy * yawTorque + ry * pitchTorque + fy * rollTorque
        local tz = uz * yawTorque + rz * pitchTorque + fz * rollTorque
        
        -- Minimal coordinated turn force (Lift redirection when banked)
        if abs(data.smoothedBankAngle) > 10 then
            local bankRadians = rad(data.smoothedBankAngle)
            -- Lift acts along the Right vector when banked
            local liftFactor = sin(bankRadians) * data.thrustForce * CONFIG.BANK_LIFT_FACTOR
            handle:AddForce(rx * liftFactor, 0, rz * liftFactor)
        end
        
        pcall(function() 
            local blocks = handle:GetBlocks() 
            if blocks and #blocks > 0 and blocks[1] and blocks[1]:Exists() then 
                blocks[1]:AddTorque(tx, ty, tz) 
            end 
        end)
    end
    
    local function applyGroundPhysics(data, currentPos, targetPos, id)
        local forward = data.handle:Forward()
        local fx, fz = forward.x, forward.z
        
        local dx, dz = targetPos.x - currentPos.x, targetPos.z - currentPos.z
        local ttx, _, ttz = normalizeXYZ(dx, 0, dz)

        -- Apply thrust
        data.handle:AddForce(fx * data.thrustForce, forward.y * data.thrustForce, fz * data.thrustForce)
        
        -- Steering (Cross product Y)
        -- Cross(Fwd, ToTgt).y
        local steeringAmount = (fz * ttx - fx * ttz) * data.steeringTorque
        
        pcall(function() 
            local blocks = data.handle:GetBlocks() 
            if blocks and #blocks > 0 and blocks[1] and blocks[1]:Exists() then 
                blocks[1]:AddTorque(0, steeringAmount, 0) 
            end 
        end)
    end
    
    -- =================================================================
    -- PUBLIC API
    -- =================================================================
    
    function PhysicsLib.spawnStructureWithConstantForce(options) 
        if not (options and options.playerId and options.blueprintName and options.structureId and options.position and options.rotation and options.force) then 
            Logger.error("Invalid constant force options") 
            return nil 
        end 
        if options.blueprintName and options.blueprintName ~= "" then
             pcall(tm.physics.AddTexture, options.blueprintName .. ".png", options.blueprintName) 
        end
        tm.players.SpawnStructure(options.playerId, options.blueprintName, options.structureId, options.position, options.rotation) 
        TimerLib.after({
            duration = 0.2, 
            onComplete = function() 
                local st = tm.players.GetSpawnedStructureById(options.structureId) 
                if st and #st > 0 and st[1] then 
                    structuresWithConstantForce[options.structureId] = { 
                        handle = st[1], 
                        force = options.force, 
                        forceMode = options.forceMode or "force", 
                        isLocalForce = options.isLocalForce or false 
                    } 
                end 
            end
        }) 
        return options.structureId 
    end
    
    function PhysicsLib.spawnStructureWithPath(options)
        if not options or not options.playerId or not options.blueprintName or not options.structureId or not options.position or not options.rotation or not options.path or not options.thrustForce or not options.steeringTorque then 
            Logger.error("Invalid path options")
            return nil 
        end
        if options.blueprintName and options.blueprintName ~= "" then
            pcall(tm.physics.AddTexture, options.blueprintName .. ".png", options.blueprintName)
        end
        tm.players.SpawnStructure(options.playerId, options.blueprintName, options.structureId, options.position, options.rotation)
        TimerLib.after({
            duration = 0.2,
            onComplete = function()
                local structureTable = tm.players.GetSpawnedStructureById(options.structureId)
                if structureTable and #structureTable > 0 and structureTable[1] then
                    local finalPath = generateBezierPath(options.path, options.loop or false, 20, false)
                    structuresFollowingPath[options.structureId] = {
                        handle = structureTable[1], 
                        path = finalPath, 
                        thrustForce = options.thrustForce,
                        steeringTorque = options.steeringTorque, 
                        loop = options.loop or false,
                        health = initHealthTracking(), 
                        playerId = options.playerId, 
                        blueprintName = options.blueprintName,
                        vehicleType = options.vehicleType or "ground", 
                        currentWaypoint = 21, 
                        pathTolerance = options.pathTolerance or 20.0,
                        currentSpeedMultiplier = 1.0, 
                        completed = false, 
                        awaitingRespawn = false,
                        yawPID = createPIDController(0.4, 0.01, 0.2, 1),
                        pitchPID = createPIDController(0.5, 0.05, 0.3, 1),
                        targetBankAngle = 0,
                        smoothedBankAngle = 0
                    }
                end
            end
        })
        return options.structureId
    end
    
    function PhysicsLib.removeConstantForce(id) 
        if structuresWithConstantForce[id] then 
            structuresWithConstantForce[id] = nil 
            return true 
        end 
        return false 
    end
    
    function PhysicsLib.removePathFollowing(id) 
        if structuresFollowingPath[id] then 
            structuresFollowingPath[id] = nil 
            return true 
        end 
        return false 
    end
    
    function PhysicsLib.cleanupAll() 
        local cleaned = 0 
        for id,_ in pairs(structuresWithConstantForce) do 
            tm.players.DespawnStructure(id) 
            cleaned = cleaned + 1 
        end 
        for id,_ in pairs(structuresFollowingPath) do 
            tm.players.DespawnStructure(id) 
            cleaned = cleaned + 1 
        end 
        structuresWithConstantForce, structuresFollowingPath = {}, {} 
        return cleaned 
    end
    
    -- =================================================================
    -- MAIN UPDATE LOOP
    -- =================================================================
    
    function PhysicsLib.update()
        if not next(structuresWithConstantForce) and not next(structuresFollowingPath) then return end
        
        for id, data in pairs(structuresWithConstantForce) do
            if data and data.handle then
                pcall(function() data.handle:AddForce(data.force.x, data.force.y, data.force.z) end)
            end
        end
    
        for id, data in pairs(structuresFollowingPath) do
            if data.awaitingRespawn then
                local st = tm.players.GetSpawnedStructureById(id)
                if st and #st > 0 and st[1] then 
                    data.handle = st[1]
                    data.awaitingRespawn = false 
                end
            end
            
            if data and data.handle and not data.completed and not data.awaitingRespawn then
                local success, pos = pcall(function() return data.handle:GetPosition() end)
                if not success then
                    data.awaitingRespawn = true
                    data.handle = nil
                else
                    local needsRecovery, reason = checkHealth(data, pos)
                    if needsRecovery then
                        recoverVehicle(id, data, reason)
                    else
                        local pathSize = #data.path
                        data.currentWaypoint = advancePathProgress(data, pos)
    
                        local targetIndex = data.currentWaypoint + CONFIG.LOOK_AHEAD_POINTS
                        
                        if targetIndex > pathSize then
                            if data.loop then
                                targetIndex = (targetIndex - 1) % pathSize + 1
                            else
                                targetIndex = pathSize
                                if dist(pos, data.path[pathSize]) < data.pathTolerance then
                                    data.completed = true
                                end
                            end
                        end
    
                        if not data.completed then
                            local targetPos = data.path[targetIndex]
                            if targetPos then
                                if data.vehicleType == "air" then
                                    applyAirPhysics(data, pos, targetPos, id)
                                else 
                                    applyGroundPhysics(data, pos, targetPos, id)
                                end
                            end
                        end
                    end
                end
            end
        end
    end
    
    UpdateManager.register(PhysicsLib.update, 700)
    return PhysicsLib
end)()

-- =================================================================
-- MODULE: DIALOGUE LIB
-- =================================================================
_G.DialogueLib = (function()
    local DialogueLib = {}
    -- tm.os.SetModTargetDeltaTime(1/60) -- Global setting

    local PlayerLib = _G.PlayerLib
    local InputLib = _G.InputLib
    local UILib = _G.UILib
    local TimerLib = _G.TimerLib
    local SceneLib = _G.SceneLib
    local SoundLib = _G.SoundLib

    local dialogues = {}
    local activePlayerDialogues = {}
    local pendingDynamicTriggers = {} -- Stores triggers waiting for an object to spawn

    local DIALOGUE_CONTEXT = "inDialogue"
    local TYPEWRITER_SPEED = 0.1

    -- =================================================================
    -- PRIVATE FUNCTIONS
    -- =================================================================


    local function drawDialogue(playerId, isNewNode)
        local state = activePlayerDialogues[playerId]
        if not state then return end

        local dialogue = state.dialogue
        local node = dialogue.nodes[state.currentNodeKey]
        if not node then
            DialogueLib.endDialogue({ target = playerId })
            return
        end

        local headerText = truncate(state.currentHeaderText)
        local message = ""

        if state.isTextScrolling then
            message = ""
        elseif node.options and #node.options > 0 then
            local option = node.options[state.selectedIndex]
            local scrollIndicator = " (" .. state.selectedIndex .. "/" .. #node.options .. ")"
            local maxOptionLength = 32 - #scrollIndicator
            local optionText = truncate(option.text, maxOptionLength)
            message = "→ " .. optionText .. scrollIndicator
        else
            message = "(Press Enter to continue)"
        end

        if isNewNode then
            UILib.createSubtleMessage({ id = "dialogue_ui", target = playerId, header = headerText, message = message, options = { icon = dialogue.icon } })
            state.isNewNode = false
        else
            UILib.updateSubtleMessageHeader({ id = "dialogue_ui", target = playerId, header = headerText })
            UILib.updateSubtleMessageText({ id = "dialogue_ui", target = playerId, message = message })
        end
    end

    local function startTypewriter(playerId)
        local state = activePlayerDialogues[playerId]
        if not state then return end
        if state.typewriterTimer then TimerLib.cancel(state.typewriterTimer) end

        local node = state.dialogue.nodes[state.currentNodeKey]
        if not node then return end

        -- Check if this dialogue uses word sounds
        local wordSound = state.dialogue.wordSound

        state.isTextScrolling = true
        local fullText = node.text
        local charIndex = 1

        state.typewriterTimer = TimerLib.every({
            interval = TYPEWRITER_SPEED,
            onTick = function()
                if not activePlayerDialogues[playerId] then TimerLib.cancel(state.typewriterTimer); return end
                if charIndex > #fullText then
                    state.isTextScrolling = false
                    drawDialogue(playerId, false)
                    TimerLib.cancel(state.typewriterTimer)
                    state.typewriterTimer = nil
                    return
                end
                state.currentHeaderText = fullText:sub(1, charIndex)

                -- Logic to play sound on new word
                if wordSound then
                    local currentChar = fullText:sub(charIndex, charIndex)
                    local prevChar = (charIndex > 1) and fullText:sub(charIndex - 1, charIndex - 1) or " "

                    if prevChar == " " and currentChar ~= " " then
                        local playerTransform = tm.players.GetPlayerTransform(playerId)
                        if playerTransform then
                            SoundLib.playAtPosition({
                                sound = wordSound,
                                position = playerTransform:GetPositionWorld()
                            })
                        end
                    end
                end

                drawDialogue(playerId, false)
                charIndex = charIndex + 1
            end
        })
    end

    local function selectOption(playerId)
        local state = activePlayerDialogues[playerId]
        if not state then return end

        if state.isTextScrolling then
            TimerLib.cancel(state.typewriterTimer)
            state.typewriterTimer = nil
            state.isTextScrolling = false
            state.currentHeaderText = state.dialogue.nodes[state.currentNodeKey].text
            drawDialogue(playerId, false)
        end

        local dialogue = state.dialogue
        local node = dialogue.nodes[state.currentNodeKey]
        if not node then return end

        if node.onLeave and type(node.onLeave) == "function" then
            pcall(node.onLeave, playerId)
        end

        local function advanceToNode(key)
            state.currentNodeKey = key
            state.selectedIndex = 1
            if dialogue.nodes[state.currentNodeKey] and dialogue.nodes[state.currentNodeKey].onEnter then
                pcall(dialogue.nodes[state.currentNodeKey].onEnter, playerId)
            end
            drawDialogue(playerId, true)
            startTypewriter(playerId)
        end

        if node.options and #node.options > 0 then
            local selectedOption = node.options[state.selectedIndex]
            if selectedOption then
                if selectedOption.onSelect then pcall(selectedOption.onSelect, playerId) end
                if selectedOption.leadsTo then advanceToNode(selectedOption.leadsTo) else DialogueLib.endDialogue({ target = playerId }) end
            end
        elseif node.leadsTo then
            advanceToNode(node.leadsTo)
        else
            DialogueLib.endDialogue({ target = playerId })
        end
    end

    local function createSingleTrigger(dialogueId, triggerData, position)
        local dialogue = dialogues[dialogueId]
        if not dialogue then return end

        local triggerOptions = {
            id = triggerData.id or ("dialogue_trigger_" .. dialogueId),
            position = position,
            size = triggerData.size or tm.vector3.Create(5, 5, 5),
            isVisible = triggerData.isVisible,
            oneTime = triggerData.oneTime,
            onEnter = function(playerId)
                DialogueLib.start({ target = playerId, dialogueId = dialogueId })
            end
        }

        if triggerData.hasExitCallback then
            triggerOptions.onExit = function(playerId)
                local state = DialogueLib.getActiveDialogue(playerId)
                if state and state.id == dialogueId then
                    DialogueLib.endDialogue({
                        target = playerId,
                        message = { header = "You walked away", text = "The conversation has ended.", icon = "UI_Icon_Warning" }
                    })
                end
            end
        end
        SceneLib.createTrigger(triggerOptions)
    end

    -- =================================================================
    -- PUBLIC API
    -- =================================================================

    function DialogueLib.create(options)
        if not options or not options.id or not options.nodes or not options.root then
            Logger.error("create requires id, nodes, and a root node key.")
            return
        end
        
        dialogues[options.id] = options

        if options.trigger then
            if options.trigger.position then
                -- Static triggers are created immediately during initialization
            elseif options.trigger.attachToId then
                -- Dynamic triggers are added to a pending list to wait for the object to spawn
                pendingDynamicTriggers[options.trigger.attachToId] = options.id
            end
        end
    end

    function DialogueLib.initialize()
        for id, dialogue in pairs(dialogues) do
            if dialogue.trigger and dialogue.trigger.position then
                createSingleTrigger(id, dialogue.trigger, dialogue.trigger.position)
            end
        end
        
        -- EVENT DRIVEN TRIGGER CREATION
        SceneLib.onObjectSpawned(function(spawnedHandle)
             if spawnedHandle and spawnedHandle.id and pendingDynamicTriggers[spawnedHandle.id] then
                local dialogueId = pendingDynamicTriggers[spawnedHandle.id]
                local dialogue = dialogues[dialogueId]
                
                Logger.info("Detected spawn of '"..spawnedHandle.id.."'. Creating trigger for dialogue '"..dialogueId.."'.")
                
                local pos = spawnedHandle:GetTransform():GetPositionWorld()
                createSingleTrigger(dialogueId, dialogue.trigger, pos)
                
                pendingDynamicTriggers[spawnedHandle.id] = nil
            end
        end)
    end

    function DialogueLib.start(options)
        if not options or not options.target or not options.dialogueId then return end
        local dialogue = dialogues[options.dialogueId]
        if not dialogue then return end

        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            local player = PlayerLib.getPlayer(pid)
            if player:getState() == "default" then
                if _G.EventLib then _G.EventLib.emit("DialogueStarted", pid, options.dialogueId) end
                player:setState(DIALOGUE_CONTEXT)
                InputLib.setContext({ target = pid, context = DIALOGUE_CONTEXT })
                activePlayerDialogues[pid] = { dialogue = dialogue, currentNodeKey = dialogue.root, selectedIndex = 1, isTextScrolling = false, currentHeaderText = "" }
                
                -- Camera Focus Logic
                if options.lookAt then
                    -- Use CameraLib to focus on the speaker
                    if CameraLib then
                        local offset = tm.vector3.Create(0, 1.5, 3.5) -- Default fallback
                        
                        -- If looking at an object, calculate offset relative to its "Front" (Forward vector)
                        if type(options.lookAt) == "table" and options.lookAt.GetTransform then
                            local tf = options.lookAt:GetTransform()
                            local fwd = tf:Forward() -- Assuming Z+ is forward
                            local up = tm.vector3.Up()
                            
                            -- Position camera in front (Forward * distance) + Up
                            -- Adjust distance (3.5) and height (1.5) as needed
                            local fwdScaled = tm.vector3.op_Multiply(fwd, 3.5)
                            local upScaled = tm.vector3.op_Multiply(up, 1.5)
                            offset = tm.vector3.op_Addition(fwdScaled, upScaled)
                        end

                        CameraLib.lookAt({ target = pid, lookAtTarget = options.lookAt, duration = 0, smoothing = 2.0, positionOffset = offset })
                    end
                end

                if dialogue.nodes[dialogue.root] and dialogue.nodes[dialogue.root].onEnter then pcall(dialogue.nodes[dialogue.root].onEnter, pid) end
                drawDialogue(pid, true)
                startTypewriter(pid)
            end
        end
    end

    function DialogueLib.endDialogue(options)
        if not options or not options.target then return end
        for _, pid in ipairs(PlayerLib.getTargetPlayers(options.target)) do
            local state = activePlayerDialogues[pid]
            if state then
                -- Get the current node and trigger its onLeave before ending.
                local currentNode = state.dialogue.nodes[state.currentNodeKey]
                if currentNode and currentNode.onLeave and type(currentNode.onLeave) == "function" then
                    pcall(currentNode.onLeave, pid)
                end

                if state.typewriterTimer then TimerLib.cancel(state.typewriterTimer) end
                local player = PlayerLib.getPlayer(pid)
                if player:isInState(DIALOGUE_CONTEXT) then player:setState("default") end
                InputLib.setContext({ target = pid, context = "default" })
                UILib.removeSubtleMessage({ id = "dialogue_ui", target = pid })
                
                -- Release Camera
                if CameraLib then CameraLib.release({ target = pid }) end

                if _G.EventLib then _G.EventLib.emit("DialogueEnded", pid) end

                activePlayerDialogues[pid] = nil
                if options.message then
                    UILib.createSubtleMessage({ target = pid, header = options.message.header, message = options.message.text, options = { duration = 2.5, icon = options.message.icon }})
                end
            end
        end
    end

    function DialogueLib.getActiveDialogue(playerId)
        local state = activePlayerDialogues[playerId]
        return state and { id = state.dialogue.id, currentNodeKey = state.currentNodeKey } or nil
    end

    -- =================================================================
    -- INPUT HANDLING
    -- =================================================================

    InputLib.onKeyDown("up", function(playerId)
        local state = activePlayerDialogues[playerId]
        if not state or state.isTextScrolling then return end
        local node = state.dialogue.nodes[state.currentNodeKey]
        if not node or not node.options or #node.options == 0 then return end
        state.selectedIndex = state.selectedIndex - 1
        if state.selectedIndex < 1 then state.selectedIndex = #node.options end
        drawDialogue(playerId, false)
    end, { context = DIALOGUE_CONTEXT })

    InputLib.onKeyDown("down", function(playerId)
        local state = activePlayerDialogues[playerId]
        if not state or state.isTextScrolling then return end
        local node = state.dialogue.nodes[state.currentNodeKey]
        if not node or not node.options or #node.options == 0 then return end
        state.selectedIndex = state.selectedIndex + 1
        if state.selectedIndex > #node.options then state.selectedIndex = 1 end
        drawDialogue(playerId, false)
    end, { context = DIALOGUE_CONTEXT })

    InputLib.onKeyDown("enter", function(playerId)
        selectOption(playerId)
    end, { context = DIALOGUE_CONTEXT })

    return DialogueLib
end)()

-- =================================================================
-- MODULE: TRAILBRIDGE LIB
-- =================================================================
_G.TrailBridge = (function()
    local TrailBridge = {}
    
    local json = _G.json
    local UpdateManager = _G.UpdateManager

    -- --- Properties ---
    TrailBridge.DEBUG = false
    TrailBridge.RECEIVE_OWN_MESSAGES = false

    local OUTBOX_FILE = "outbox.json"
    local INBOX_FILE = "inbox.json"
    local STATE_FILE = "inbox.state.json"
    local CHECK_INTERVAL = 0.2
    local MOD_TAG = "\u{2060}"

    local localSessionId = nil
    local localPlayerName = nil
    local checkTimer = 0
    local outboxQueue = {}
    local callbacks = {}
    local pendingQueries = {}
    local queryCounter = 0
    local lastProcessedTimestamp = 0
    local isInitialized = false

    -- --- Helper Functions ---
    local function log(message)
        if TrailBridge.DEBUG then
            Logger.debug("[TrailBridge] " .. tostring(message))
        end
    end

    local function readQueueFile(filename)
        local success, fileContent = pcall(tm.os.ReadAllText_Dynamic, filename)
        if not success or not fileContent or fileContent == "" then return {} end
        local parseSuccess, data = pcall(json.parse, fileContent)
        if not parseSuccess then
            -- Only log parsing errors, not missing file errors (common on startup)
            if TrailBridge.DEBUG then 
                Logger.warn("Could not parse " .. filename)
            end
            return {}
        end
        return data
    end

    local function writeOutbox()
        if #outboxQueue == 0 then return end
        log("Writing " .. #outboxQueue .. " items to outbox.")
        pcall(tm.os.WriteAllText_Dynamic, OUTBOX_FILE, json.serialize(outboxQueue))
        outboxQueue = {}
    end

    local function processInbox()
        local inboxQueue = readQueueFile(INBOX_FILE)
        if #inboxQueue == 0 then return end

        local newHighestTimestamp = lastProcessedTimestamp
        local processedNewMessages = false

        for _, data in ipairs(inboxQueue) do
            local eventTimestamp = data.server_timestamp or 0
            if eventTimestamp > lastProcessedTimestamp then
                processedNewMessages = true
                if eventTimestamp > newHighestTimestamp then newHighestTimestamp = eventTimestamp end

                -- Filter own messages if needed
                if not TrailBridge.RECEIVE_OWN_MESSAGES and data.sessionId == localSessionId and data.channel ~= "db_response" then
                    goto continue
                end

                -- Handle DB Response
                if data.channel == "db_response" and data.sessionId == localSessionId then
                    if data.req_id and pendingQueries[data.req_id] then
                        log("Received DB Response for " .. data.req_id)
                        pcall(pendingQueries[data.req_id], data.response)
                        pendingQueries[data.req_id] = nil
                    end
                    goto continue
                end

                -- Dispatch Callback
                if data.channel and callbacks[data.channel] then
                    log("Dispatching callback for " .. data.channel)
                    pcall(callbacks[data.channel], data)
                end
            end
            ::continue::
        end

        if newHighestTimestamp > lastProcessedTimestamp then
            lastProcessedTimestamp = newHighestTimestamp
            local success, jsonStr = pcall(json.serialize, { last_processed_ts = newHighestTimestamp })
            if success then
                pcall(tm.os.WriteAllText_Dynamic, STATE_FILE, jsonStr)
            end
        end
    end

    -- --- Public API ---

    function TrailBridge.init()
        if isInitialized then return end
        
        -- Generate Session ID
        localSessionId = tm.os.GetRealtimeSinceStartup() .. "-" .. math.random(10000, 99999)
        
        -- Reset Outbox using pcall
        outboxQueue = {}
        pcall(tm.os.WriteAllText_Dynamic, OUTBOX_FILE, "[]")

        -- Ensure Inbox exists to prevent read errors
        local success, content = pcall(tm.os.ReadAllText_Dynamic, INBOX_FILE)
        if not success or not content or content == "" then
            Logger.info("Creating new " .. INBOX_FILE)
            pcall(tm.os.WriteAllText_Dynamic, INBOX_FILE, "[]")
        end

        -- Ensure State file exists
        local successState, stateContent = pcall(tm.os.ReadAllText_Dynamic, STATE_FILE)
        if not successState or not stateContent or stateContent == "" then
             -- No need to log this creation as it's a silent init step, or use debug log
             pcall(tm.os.WriteAllText_Dynamic, STATE_FILE, "{}")
             stateContent = "{}"
        end

        -- Load State
        if stateContent and stateContent ~= "" then
            local success, data = pcall(json.parse, stateContent)
            if success and data.last_processed_ts then
                lastProcessedTimestamp = data.last_processed_ts
            end
        end
        
        isInitialized = true
        Logger.debug("TrailBridge Initialized. Session: " .. localSessionId)
    end
    
    local function ensureInit()
        if not isInitialized then TrailBridge.init() end
    end

    function TrailBridge.update()
        ensureInit()
        local dt = tm.os.GetModDeltaTime()
        checkTimer = checkTimer + dt
        if checkTimer >= CHECK_INTERVAL then
            checkTimer = 0
            processInbox()
            writeOutbox()
        end
    end

    function TrailBridge.send(channel, data)
        ensureInit()
        if not channel or data == nil then return end

        local payload = (type(data) == "table") and data or { value = data }
        payload.channel = channel
        payload.sessionId = localSessionId
        if localPlayerName then payload.sender = localPlayerName end
        
        table.insert(outboxQueue, payload)
    end
    
    function TrailBridge.on(channel, callback)
        ensureInit()
        if channel and type(callback) == "function" then
            callbacks[channel] = callback
        end
    end

    -- DB Helpers
    local function query(payload, callback)
        ensureInit() 
        if not payload or not callback then return end
        queryCounter = queryCounter + 1
        local req_id = localSessionId .. "_" .. queryCounter
        pendingQueries[req_id] = callback
        TrailBridge.send("db_query", { req_id = req_id, payload = payload })
    end

    function TrailBridge.setData(apiKey, key, value, callback)
        ensureInit()
        if not apiKey or not key or value == nil or not callback then return end
        query({ action = "set", api_key = apiKey, key = key, value = value }, callback)
    end

    function TrailBridge.getData(apiKey, key, callback)
        ensureInit()
        if not apiKey or not key or not callback then return end
        query({ action = "get", api_key = apiKey, key = key }, callback)
    end
    
    -- Chat Helpers
    function TrailBridge.getTaggedSender(name)
        return name .. MOD_TAG
    end
    
    function TrailBridge.handleChatMessage(senderName, message)
        ensureInit()
        if string.find(senderName, MOD_TAG) then return true end
        localPlayerName = senderName
        TrailBridge.send("chat", { message = message, timestamp = tm.os.GetTime() })
        return false
    end

    -- Register Update
    if UpdateManager then
        UpdateManager.register(TrailBridge.update, 500)
    end

    return TrailBridge
end)()

-- Public API
_G.TrailKit = {
    Player   = _G.PlayerLib,
    UI       = _G.UILib,
    Scene    = _G.SceneLib,
    Dialogue = _G.DialogueLib,
    Input    = _G.InputLib,
    Timer    = _G.TimerLib,
    Event    = _G.EventLib,
    Save     = _G.SaveLib,
    Math     = _G.MathLib,
    Camera   = _G.CameraLib,
    Sound    = _G.SoundLib,
    Physics  = _G.PhysicsLib,
    Game     = _G.Game,
    Perms    = _G.PermissionsLib,
    Bridge   = _G.TrailBridge,
    Log      = _G.Logger
}

UpdateManager.initialize()

return UpdateManager