-- =================================================================
-- WELCOME TO YOUR MOD SCRIPT!
-- =================================================================
-- This "main.lua" file is the heart of your mod. It tells the game what to do.

-- First, we load TrailKit. 
-- Think of this as opening a toolbox. It gives us powerful tools so we don't have to write everything from scratch.
tm.os.DoFile("trailkit")

-- =================================================================
-- 1. DIALOGUE SYSTEM
-- =================================================================
-- This function is where we write scripts for NPCs (Non-Player Characters).
-- A dialogue is like a tree: it has a "root" (start) and branches into "nodes".

function defineDialogues()
    -- Example 1: An interactive Quest Giver
    TrailKit.Dialogue.create({
        id = "npc_quest_chat",         -- A unique name for this dialogue
        icon = "info",         -- The icon to show above the text
        wordSound = "UI_IntercomMsg_Show", -- Sound to play while text is typing (Chirp)
        root = "start",                -- We start at the node named "start"
        
        -- "nodes" is a list of all the different things the NPC can say
        nodes = {
            -- The starting message
            start = {
                text = "Hello, adventurer! I need your help.",
                -- "options" are menu options
                options = {
                    { text = "How can I help you?", leadsTo = "quest_details" }, -- Jumps to "quest_details"
                    { text = "I'm not interested.", leadsTo = "end_busy" }       -- Jumps to "end_busy"
                }
            },
            -- The explanation node
            quest_details = {
                text = "My score is 0! Please click the blue crystal to give me a point.",
                options = {
                    { text = "I will do it!", leadsTo = "end_accept" },
                    { text = "No way!", leadsTo = "end_busy" }
                }
            },
            -- Ending nodes (no options, so the chat closes after this)
            end_accept = { text = "Thank you so much!" },
            end_busy = { text = "A shame. Come back if you reconsider." }
        }
    })

    -- Example 2: A "Monologue" (NPC talks to themselves, or narrates)
    -- This one also has a "trigger", creating a magical zone in the world automatically.
    TrailKit.Dialogue.create({
        id = "monologue_chat",
        icon = "warning",
        root = "line1",
        nodes = {
            line1 = { text = "The crystal glows...", leadsTo = "line2" }, -- jumps to line2 immediately when done
            line2 = { text = "What secrets does it hold?", leadsTo = "line3" },
            line3 = { text = "Measure carefully." }
        },
        -- This creates a physical invisible box in the world. walking into it starts the chat.
        trigger = {
            position = tm.vector3.Create(285, 220, 915),
            size = tm.vector3.Create(5, 5, 5),
            isVisible = true,
            oneTime = true -- Only happens once per player
        }
    })

    -- Example 3: A Blacksmith with simple lines
    TrailKit.Dialogue.create({
        id = "blacksmith_monologue",
        icon = "info",
        root = "line1",
        nodes = {
            line1 = { text = "Busy forging here.", leadsTo = "line2" },
            line2 = { text = "The crystal's energy is potent.", leadsTo = "line3" },
            line3 = { text = "Make good use of it." }
        }
    })

    -- Example 4: Using Code inside Dialogue (Advanced!)
    -- You can make things happen (like spawning items) when a player reads a specific line.
    TrailKit.Dialogue.create({
        id = "stone_keeper_chat",
        icon = "info",
        wordSound = "UI_IntercomMsg_Show",
        root = "start",
        nodes = {
            start = {
                text = "Behold the magic stone!",
                -- onEnter runs when this specific text box appears
                onEnter = function(playerId)
                    local npc = TrailKit.Scene.getObject("npc_stone_keeper")
                    if npc and npc:Exists() then
                        -- Math to find the spot in front of the NPC
                        local npcTransform = npc:GetTransform()
                        local npcPos = npcTransform:GetPositionWorld()
                        local npcForward = npcTransform:Forward()
                        local stonePos = tm.vector3.op_Addition(npcPos, tm.vector3.op_Multiply(npcForward, 2))
                        stonePos.y = npcPos.y + 0.5 
                        
                        -- Spawn a stone and apply an animation to it
                        local stone = TrailKit.Scene.spawnObject({ name = "PFB_SimpleStoneHighSeas", id = "magic_stone", position = stonePos, scale = 0.1 })
                        if stone then stone:bob({ height = 0.2, speed = 1, loop = true }) end
                    end
                end,
                -- onLeave runs when the player closes this chat or moves to a node that isn't this one
                onLeave = function(playerId)
                    local stone = TrailKit.Scene.getObject("magic_stone")
                    if stone then stone:despawn() end -- Poof! Stone gone.
                end,
                options = {
                    { text = "That's impressive!", leadsTo = "end_node" },
                    { text = "I must be going." }
                }
            },
            end_node = {
                text = "Indeed. It is MY precious."
            }
        }
    })
end

-- =================================================================
-- 2. PERMISSIONS & VARIABLES
-- =================================================================

-- We define "roles" for players. 
-- "user" (Level 1) is registered by default.
-- We can add more roles, like "admin" (Level 2).
TrailKit.Perms.registerRole({ name = "admin", level = 2 })

-- "Variables" are containers for storing information we need later.
-- We start them as 'nil' (empty) and fill them when the game starts.
local clickableObject = nil
local sequenceObject = nil
local vehicle = nil
local turret = nil
local turretBarrel = nil
local isVehicleMoving = false
local npc = nil
local blacksmithNpc = nil
local stoneKeeperNpc = nil

-- =================================================================
-- 3. MENU DEFINITIONS (UI)
-- =================================================================

-- Definition for a window style menu
local windowMenuDefinition = {
    title = "Main Menu (Window)",
    items = {
        {
            type = "button",
            text = "Give Jetpack",
            -- This function runs when the button is clicked
            onClick = function(data)
                local player = TrailKit.Player.getPlayer(data.playerId)
                if player then
                    player:setJetpack(true) -- Turn on jetpack!
                    -- Show a small message
                    TrailKit.UI.createSubtleMessage({ target = data.playerId, header = "Jetpack", message = "Jetpack Enabled!", options = { duration = 2 }})
                end
            end
        },
        {
            type = "button",
            text = "Close",
            action = "back" -- Special action to close/go back
        }
    }
}

-- Definition for a "Subtle" menu (Small notification at bottom right)
-- This is great for admin panels or quick actions.
local subtleMenuDefinition = {
    style = "subtle",
    icon = "menu",
    root = "main", -- Starts at the 'main' list
    menus = {
        main = {
            title = "Main Menu",
            options = {
                -- These options open other lists (submenus)
                { name = "Fun Actions", action = "submenu", submenu = "actions" },
                { name = "Settings", action = "submenu", submenu = "settings" },
                -- Only show this if player has "admin" permission!
                { name = "Admin Panel", action = "submenu", submenu = "admin_panel", permission = "admin" },
                { name = "Close Menu", action = "back" }
            }
        },
        actions = {
            title = "Fun Actions",
            -- dynamic options lets us create the list with code (e.g. valid actions only)
            getDynamicOptions = function(context)
                local options = {}
                local player = TrailKit.Player.getPlayer(context.playerId)
                if player then
                    table.insert(options, { name = "Give Jetpack", action = "execute", onExecute = function()
                        player:setJetpack(true)
                        TrailKit.UI.createSubtleMessage({ target = context.playerId, header = "Jetpack", message = "Jetpack Enabled!", options = { duration = 2 }})
                    end })
                end
                return options
            end
        },
        settings = {
            title = "Settings",
            options = {
                {
                    name = "God Mode",
                    action = "toggle", -- A checkbox style option
                    -- Check current state
                    getValue = function(context)
                        return TrailKit.Player.getPlayer(context.playerId):getData("godMode")
                    end,
                    -- Change state
                    onToggle = function(context)
                        local p = TrailKit.Player.getPlayer(context.playerId)
                        local current = p:getData("godMode")
                        p:setData("godMode", not current)
                        p:setInvincible(not current)
                        TrailKit.UI.createSubtleMessage({ target = context.playerId, header = "Settings", message = "God mode " .. ((not current) and "enabled" or "disabled"), options = { duration = 2 }})
                    end
                },
                {
                    name = "Game Speed",
                    action = "adjust", -- A slider/left-right style option
                    getValue = function() return string.format("%.1f", TrailKit.Game.getTimeScale()) .. "x" end,
                    onAdjust = function(context, direction)
                        local newSpeed = TrailKit.Game.getTimeScale() + (0.1 * direction)
                        TrailKit.Game.setTimeScale(math.max(0.1, math.min(2.0, newSpeed)))
                    end
                }
            }
        },
        admin_panel = {
            title = "Admin Panel",
            options = {
                { name = "Kick Player", action = "submenu", submenu = "kick_player_list" }
            }
        },
        kick_player_list = {
            title = "Select Player to Kick",
            -- Dynamically list all players on the server
            getDynamicOptions = function(context)
                local options = {}
                for id, player in pairs(TrailKit.Player.getAllPlayers()) do
                    if id ~= context.playerId then -- Don't let me kick myself (although impossible)
                        table.insert(options, {
                            name = player.name,
                            action = "execute",
                            onExecute = function()
                                tm.players.Kick(player.id)
                                TrailKit.Input.broadcast("Admin kicked " .. player.name .. ".", tm.color.Yellow())
                                TrailKit.UI.closeMenu({ target = context.playerId })
                            end
                        })
                    end
                end
                return options
            end
        }
    }
}

-- =================================================================
-- 4. GAMEPLAY FUNCTIONS
-- =================================================================

-- Updates the custom HUD text for a specific player
function UpdateHUD(playerObject)
    local score = playerObject:getData("score") or 0
    local state = playerObject:getState()
    
    -- updateSubtleMessageText changes the text of an existing message without recreating it
    TrailKit.UI.updateSubtleMessageText({
        id = "player_hud",
        target = playerObject.id,
        message = "Score: " .. tostring(score) .. "  |  Status: " .. state
    })
end

-- Registers commands that players can type in chat (e.g., /kick, /help)
function RegisterChatCommands()
    TrailKit.Input.registerCommand({
        name = "kick",
        permission = "admin", -- Only admins can use this
        description = "Kicks a player from the server.",
        callback = function(caller, args)
            if not args[2] then
                caller:message("Usage: /kick <player_name>", tm.color.Gray())
                return
            end
            local targetPlayer = TrailKit.Player.getPlayerByName(args[2])
            if targetPlayer then
                if targetPlayer.id == caller.id then
                    caller:message("You cannot kick yourself.", tm.color.Red())
                    return
                end
                tm.players.Kick(targetPlayer.id)
                TrailKit.Input.broadcast(caller.name .. " kicked " .. targetPlayer.name .. ".", tm.color.Yellow())
            else
                caller:message("Player not found: " .. args[2], tm.color.Red())
            end
        end
    })

    TrailKit.Input.registerCommand({
        name = "godmode",
        permission = "admin",
        description = "Toggles invincibility for a player.",
        callback = function(caller, args)
            local targetName = args[2] or caller.name
            local targetPlayer = TrailKit.Player.getPlayerByName(targetName)
            if targetPlayer then
                local current = targetPlayer:getData("godMode")
                targetPlayer:setData("godMode", not current)
                targetPlayer:setInvincible(not current)
                local status = (not current) and "enabled" or "disabled"
                TrailKit.Input.broadcast("God mode " .. status .. " for " .. targetPlayer.name, tm.color.Yellow())
            else
                caller:message("Player not found: " .. targetName, tm.color.Red())
            end
        end
    })

    TrailKit.Input.registerCommand({
        name = "setscore",
        permission = "admin",
        description = "Sets the score for a player.",
        callback = function(caller, args)
            if not args[2] or not args[3] then
                caller:message("Usage: /setscore <player_name> <score>", tm.color.Gray())
                return
            end
            local targetPlayer = TrailKit.Player.getPlayerByName(args[2])
            local score = tonumber(args[3])
            if targetPlayer and score then
                targetPlayer:setData("score", score)
                UpdateHUD(targetPlayer)
                TrailKit.Input.broadcast(targetPlayer.name .. "'s score set to " .. score, tm.color.Yellow())
            else
                caller:message("Invalid player or score.", tm.color.Red())
            end
        end
    })

    TrailKit.Input.registerCommand({
        name = "help",
        permission = "user",
        description = "Shows a list of available commands.",
        callback = function(caller, args)
            TrailKit.Input.messagePlayer({ player = caller, message = "--- Available Commands ---", color = tm.color.Cyan() })
            local availableCommands = TrailKit.Input.getAvailableCommands(caller)
            for _, cmd in ipairs(availableCommands) do
                caller:message("/" .. cmd.name .. " - " .. cmd.description, tm.color.White())
            end
        end
    })
end

-- =================================================================
-- 5. INTERACTION LOGIC (What happens when you click things!)
-- =================================================================

-- This is called when the player presses '1' or clicks the main crystal
function OnClickObject(playerId)
    local playerObject = TrailKit.Player.getPlayer(playerId)
    if playerObject:getState() ~= "default" then return end -- Don't run if busy
    if not clickableObject or not clickableObject:Exists() then return end

    -- Increase score
    local currentScore = playerObject:getData("score") or 0
    playerObject:setData("score", currentScore + 1)
    playerObject:save() -- Save to disk immediately
    UpdateHUD(playerObject)

    -- Fun visuals on the crystal
    clickableObject:pulse({ loop = false, maxScale = 1.3, speed = 5 }) -- Grow/Shrink
    
    -- Play sound
    clickableObject:stopAllSounds()
    clickableObject:playSound("UI_Generic_positiveFeedback") 
    
    playerObject:toast("Score Increased!", 1.5)

    -- Spawn some small effects (Scrap)
    for i = 1, 8 do
        local scrap = TrailKit.Scene.getFromPool({ id = "scrap_pool", spawnOptions = { position = clickableObject:GetTransform():GetPositionWorld() }})
        if scrap then
            -- Fling them in random directions
            local randomForce = tm.vector3.Create(math.random(-10, 10), math.random(5, 15), math.random(-10, 10))
            scrap:applyForce(tm.vector3.op_Multiply(randomForce, 30))
        end
    end

    -- Trigger a Cutscene if score is 5
    if playerObject:getData("score") == 5 then
        TrailKit.UI.createSubtleMessage({ target = playerId, header = "Score Reached!", message = "You reached 5 points! Here's a cinematic.", options = { duration = 4.0 }})
        StartCutscene(playerId)
    end
end

-- Starts a cinematic sequence
function StartCutscene(playerId)
    local playerObject = TrailKit.Player.getPlayer(playerId)
    if playerObject:getState() ~= "default" then return end -- Busy check

    if not clickableObject or not clickableObject:Exists() then return end

    -- Change state to "inCutscene" for 5 seconds (blocks movement/input)
    playerObject:setState("inCutscene", 5.0)
    ShowcaseCamera(playerId)
end

-- Moves the camera to look at the crystal
function ShowcaseCamera(playerId)
    local playerObject = TrailKit.Player.getPlayer(playerId)
    if not clickableObject or not clickableObject:Exists() then return end
    
    -- Smoothly pan camera to position [0, 3, -8] relative to the crystal
    playerObject:lookAt(clickableObject, {
        duration = 5.0,
        smoothing = 2.0,
        positionOffset = tm.vector3.Create(0, 3, -8)
        -- We don't define 'target', so it looks at the object center
    })
end

-- Finds all red containers and pulses them (Showcase feature)
function ShowcaseQueries(playerId)
    if not clickableObject or not clickableObject:Exists() then return end
    
    -- Find all objects named "PFB_Container_Red"
    local allContainers = TrailKit.Scene.findEntitiesByName("PFB_Container_Red")
    
    TrailKit.UI.createSubtleMessage({ target = playerId, header = "Query", message = "Found " .. #allContainers .. " red containers.", options = { duration = 3.0 }})
    
    -- Animate each one we found
    for _, container in ipairs(allContainers) do
        container:pulse({ loop = false, maxScale = 1.1, speed = 1.5 })
    end
end

-- Demonstartes a sequenced animation (Bob then Sway)
function ShowcaseSequence(playerId)
    if not sequenceObject or not sequenceObject:Exists() then return end
    TrailKit.UI.createSubtleMessage({ target = playerId, header = "Animation", message = "Starting Sequence!", options = { duration = 3.0 }})
    
    sequenceObject:sequence({
        animations = {
            { type = "bob", options = { height = 2.0, speed = 0.5 } },
            { type = "sway", options = { distance = 3.0, speed = 0.5 } }
        },
        loop = true,
        pingpong = true
    })
end

-- Demonstrates animating a Group (Parent + Children together)
function ShowcaseGroupAnimation(playerId)
    if not turret or not turret:Exists() then return end
    TrailKit.UI.createSubtleMessage({ target = playerId, header = "Group Animation", message = "Animating turret assembly!", options = { duration = 3.0 }})

    turret:sequence({
        delay = 0.2,
        loop = true,
        animations = {
            { type = "pulse", options = { maxScale = 1.15, speed = 0.4 } }
        }
    })
end

function ShowcaseSpin(playerId)
    if not clickableObject or not clickableObject:Exists() then return end
    TrailKit.UI.createSubtleMessage({ target = playerId, header = "Animation", message = "PingPong spin!", options = { duration = 4.0 }})
    clickableObject:spin(180, nil, { loop = false, pingpong = true })
end

function ShowcaseScaleTween(playerId)
    if not clickableObject or not clickableObject:Exists() then return end
    TrailKit.UI.createSubtleMessage({ target = playerId, header = "Animation", message = "Tweening scale!", options = { duration = 4.0 }})
    clickableObject:scale(tm.vector3.Create(0.5, 2.0, 0.5), 2.0, {
        pingpong = true,
        easeType = "easeInOutElastic",
        scaleDirection = tm.vector3.Create(0, -1, 0),
        pivotPoint = tm.vector3.Create(0, -1, 0)
    })
end

-- Spawns/Moves a vehicle in a loop
function ToggleVehicleMovement(playerId)
    if not vehicle or not vehicle:Exists() then return end
    isVehicleMoving = not isVehicleMoving
    
    if isVehicleMoving then
        TrailKit.UI.createSubtleMessage({ target = playerId, header = "Animation", message = "Starting vehicle patrol!", options = { duration = 2.0 }})
        local startPos = vehicle:GetTransform():GetPositionWorld()
        
        -- moves an object through a list of positions
        vehicle:path({
            frames = {
                { position = startPos, duration = 0 },
                { position = tm.vector3.op_Addition(startPos, tm.vector3.Create(20, 0, 0)), duration = 5.0 },
                { position = tm.vector3.op_Addition(startPos, tm.vector3.Create(20, 0, 10)), duration = 3.0 },
                { position = tm.vector3.op_Addition(startPos, tm.vector3.Create(0, 0, 10)), duration = 5.0 },
                { position = startPos, duration = 3.0 }
            },
            loop = true,
            easeType = "easeInOutSine"
        })
    else
        TrailKit.UI.createSubtleMessage({ target = playerId, header = "Animation", message = "Stopping vehicle patrol!", options = { duration = 2.0 }})
        vehicle:stopAnimation() -- Stop moving
    end
end

-- Spawns a vehicle that follows a complex path on the ground
function ShowcaseGroundPathFollowing(playerId)
    TrailKit.UI.createSubtleMessage({ target = playerId, header = "Physics", message = "Spawning a ground vehicle!", options = { duration = 4.0 }})
    local basePos = tm.vector3.Create(238.4, 215.6, 887.9)
    -- Define path points...
    local path = { basePos, tm.vector3.op_Addition(basePos, tm.vector3.Create(6,0,0)), tm.vector3.op_Addition(basePos, tm.vector3.Create(8,0,2)), tm.vector3.op_Addition(basePos, tm.vector3.Create(8,0,6)), tm.vector3.op_Addition(basePos, tm.vector3.Create(6,0,8)), tm.vector3.op_Addition(basePos, tm.vector3.Create(2,0,8)), tm.vector3.op_Addition(basePos, tm.vector3.Create(0,0,6)), tm.vector3.op_Addition(basePos, tm.vector3.Create(0,0,2)), basePos }
    
    local player = TrailKit.Player.getPlayer(playerId)
    if player then
        -- This spawns an blueprint vehicle structure
        player:spawnVehicle({
            blueprintName = "car", 
            structureId = "ground_vehicle_1",
            position = basePos, 
            rotation = tm.vector3.Create(0,0,0), 
            path = path, -- It will follow this path automatically
            thrustForce = 1, 
            steeringTorque = 600, 
            loop = true, 
            vehicleType = "ground"
        })
    end
end

-- Spawns a plane that flies in a circle
function ShowcaseAirPathFollowing(playerId)
    TrailKit.UI.createSubtleMessage({ target = playerId, header = "Physics", message = "Spawning an air vehicle! ✈️", options = { duration = 4.0 }})

    local centerPos = tm.vector3.Create(200, 405, 700)
    local radius = 650
    local altitude = centerPos.y + 20.0
    local numPoints = 24
    local path = {}

    -- Calculate circle points via math
    for i = 0, numPoints do
        local angle = (i / numPoints) * 2 * math.pi
        local x = centerPos.x + radius * math.cos(angle)
        local z = centerPos.z + radius * math.sin(angle)
        table.insert(path, tm.vector3.Create(x, altitude, z))
    end

    local player = TrailKit.Player.getPlayer(playerId)
    if player then
        player:spawnVehicle({
            blueprintName = "plane",
            structureId = "air_vehicle_1",
            position = path[1],
            rotation = tm.vector3.Create(0, 0, 0),
            path = path,
            thrustForce = 20,
            steeringTorque = 5000,
            pathTolerance = 4.0,
            loop = true,
            vehicleType = "air"
        })
    end
end

function ToggleWindowMenu(playerId)
    if TrailKit.UI.isOpen(playerId, "standard") then
        TrailKit.UI.closeMenu({ target = playerId, style = "standard" })
    else
        TrailKit.UI.showMenu({ target = playerId, definition = windowMenuDefinition })
    end
end

function ToggleSubtleMenu(playerId)
    if TrailKit.UI.isOpen(playerId, "subtle") then
        TrailKit.UI.closeMenu({ target = playerId, style = "subtle" })
    else
        TrailKit.UI.showMenu({ target = playerId, definition = subtleMenuDefinition })
    end
end

-- =================================================================
-- 6. SCENE SETUP (Spawning Objects)
-- =================================================================

function SetupGameScene()
    local mainPos = tm.vector3.Create(300, 220, 915)

    -- Only spawn if we haven't already (prevents duplicates)
    if not clickableObject or not clickableObject:Exists() then
        TrailKit.Log.info("SceneLib: Creating scene objects for the first time.")

        -- Spawn main crystal
        clickableObject = TrailKit.Scene.spawnObject({ name = "PFB_CrystalSmall_blue", position = mainPos, id = "main_crystal", scale = 0.5 })
        
        -- Spawn decoration containers
        TrailKit.Scene.spawnObject({ name = "PFB_Container_Red", position = tm.vector3.op_Addition(mainPos, tm.vector3.Create(10, 0, 5)), scale = 0.3 })
        TrailKit.Scene.spawnObject({ name = "PFB_Container_Red", position = tm.vector3.op_Addition(mainPos, tm.vector3.Create(-8, 0, -8)), scale = 0.4 })

        -- Spawn test sphere
        sequenceObject = TrailKit.Scene.spawnObject({
            name = "PFB_TestCollisionTimeline",
            position = tm.vector3.op_Addition(mainPos, tm.vector3.Create(0, 5, 15)),
            id = "sequence_sphere", scale = 0.4
        })

        -- Spawn NPC 1: Chirpo Blue
        npc = TrailKit.Scene.spawnObject({
            name = "PFB_Chirpo_Blue",
            position = tm.vector3.op_Addition(mainPos, tm.vector3.Create(5, -0.5, 10)),
            id = "npc_1"
        })
        npc:bob({ height = 0.2, speed = 0.5, loop = true }) -- Make him bob up and down

        -- Spawn NPC 2: Blacksmith
        blacksmithNpc = TrailKit.Scene.spawnObject({
            name = "PFB_Chirpo_Orange",
            position = tm.vector3.op_Addition(mainPos, tm.vector3.Create(-10, -0.5, 10)),
            id = "npc_blacksmith"
        })
        blacksmithNpc:bob({ height = 0.1, speed = 0.3, loop = true })

        -- Spawn NPC 3: Stone Keeper
        stoneKeeperNpc = TrailKit.Scene.spawnObject({
            name = "PFB_Chirpo_LightGreen",
            position = tm.vector3.op_Addition(mainPos, tm.vector3.Create(-30, -4.36, -30)),
            id = "npc_stone_keeper"
        })
        stoneKeeperNpc:pulse({ loop = true, maxScale = 1.05, minScale = 0.95, speed = 0.2 }) -- Make him breathe/pulse

        -- Complex grouping: Turret on top of a Vehicle
        local vehiclePos = tm.vector3.op_Addition(mainPos, tm.vector3.Create(15, 0, 0))
        vehicle = TrailKit.Scene.spawnObject({ name = "PFB_TestCollisionTimeline", position = vehiclePos, id = "vehicle_base", scale = tm.vector3.Create(4, 2, 6) })

        -- Create a "Group" attached to the vehicle
        turret = TrailKit.Scene.createGroup({
            id = "turret_assembly", group = vehicle,
            position = tm.vector3.op_Addition(vehiclePos, tm.vector3.Create(0, 1, 0))
        })

        -- Add parts to the group
        local turretPos = turret:GetTransform():GetPositionWorld()
        local turretBase = TrailKit.Scene.spawnObject({
            name = "PFB_TestCollisionTimeline", group = turret,
            position = turretPos, scale = tm.vector3.Create(3, 1, 3)
        })
        -- Adjust position relative to parent
        turretBase:setPosition(tm.vector3.op_Subtraction(turret:GetTransform():GetPositionWorld(), tm.vector3.Create(0, 8, 0)))

        turretBarrel = TrailKit.Scene.spawnObject({
            name = "PFB_TestCollisionTimeline", id = "turret_barrel", group = turret,
            position = turretPos, scale = tm.vector3.Create(0.6, 0.6, 4)
        })
        turretBarrel:setPosition(tm.vector3.op_Subtraction(turret:GetTransform():GetPositionWorld(), tm.vector3.Create(0, 6, 12)))

        turretBarrel:lookAt(nil, { smoothing = 0.1 }) -- Make the barrel look at world center

        -- Create a pool of objects for effects (recyclable)
        TrailKit.Scene.createPool({ id = "scrap_pool", objectName = "PFB_MovePuzzleBall", size = 20 })

        -- Load the dialogues we defined earlier
        TrailKit.Dialogue.initialize()

        -- Attach Click Listeners
        if clickableObject then
             clickableObject:onClick(function(pid) OnClickObject(pid) end)
        end
        
        if npc then
             npc:onClick(function(pid) 
                 local p = TrailKit.Player.getPlayer(pid)
                 p:startDialogue("npc_quest_chat", npc) 
             end)
        end
        
        if blacksmithNpc then
             blacksmithNpc:onClick(function(pid)
                 local p = TrailKit.Player.getPlayer(pid)
                 p:startDialogue("blacksmith_monologue", blacksmithNpc)
             end)
        end
        
        if stoneKeeperNpc then
             stoneKeeperNpc:onClick(function(pid)
                 local p = TrailKit.Player.getPlayer(pid)
                 p:startDialogue("stone_keeper_chat", stoneKeeperNpc)
             end)
        end
        
        -- Special: Find all spawned red containers and make them clickable too
        local containers = TrailKit.Scene.findEntitiesByName("PFB_Container_Red")
        for _, c in ipairs(containers) do
            c:onClick(function(pid)
                 c:pulse({ loop = false, maxScale = 1.1, speed = 4 })
                 local p = TrailKit.Player.getPlayer(pid)
                 if p then p:toast("You clicked a red container.", 2.0) end
            end)
        end
    end
end

-- =================================================================
-- 7. EVENT LISTENERS
-- =================================================================

-- Registers key bindings for testing features
function RegisterShowcaseKeys(player)
    player:onKey("1", function() OnClickObject(player.id) end)
    player:onKey("2", function() ShowcaseCamera(player.id) end)
    player:onKey("3", function() ShowcaseQueries(player.id) end)
    player:onKey("4", function() ShowcaseSequence(player.id) end)
    player:onKey("5", function() ShowcaseSpin(player.id) end)
    player:onKey("6", function() ShowcaseScaleTween(player.id) end)
    player:onKey("7", function() ToggleVehicleMovement(player.id) end)
    player:onKey("8", function() StartCutscene(player.id) end)
    player:onKey("9", function() ShowcaseGroundPathFollowing(player.id) end)
    player:onKey("0", function() ShowcaseAirPathFollowing(player.id) end)
    player:onKey("=", function() ShowcaseGroupAnimation(player.id) end)
    player:onKey("m", function() ToggleWindowMenu(player.id) end)
    player:onKey("n", function() ToggleSubtleMenu(player.id) end)
end

-- Runs when a player joins the server
TrailKit.Event.on("onPlayerJoined", function(playerObject)
    -- Grant admin to first player (ID 0)
    if playerObject.id == 0 then
        playerObject:grantPermission("admin")
    end
    
    if playerObject:hasPermission("admin") then
        TrailKit.Input.broadcast(playerObject.name .. " has joined the server (Admin).", tm.color.Cyan())
    else
        playerObject:toast(playerObject.name .. " has joined the server.", 4.0)
    end
    
    -- Load saved data
    playerObject:makePersistent()
    
    -- Setup controls
    RegisterShowcaseKeys(playerObject)
    
    -- Create the HUD (Heads Up Display)
    TrailKit.UI.createSubtleMessage({
        id = "player_hud",
        target = playerObject.id,
        header = "Player Status",
        message = "Score: 0  |  Status: default",
        options = { icon = "cog" }
    })
    TrailKit.UI.createSubtleMessage({
        id = "info_panel",
        target = playerObject.id,
        header = "Controls",
        message = "Click, Keys 1-9, M/N menus.",
        options = { duration = 15.0 }
    })
    
    -- Refresh the HUD text
    UpdateHUD(playerObject)
end)

-- Runs ONCE when the mod actually starts loading
function OnModStart()
   SetupGameScene()
end

-- Runs when a player leaves
TrailKit.Event.on("onPlayerLeft", function(playerObject)
    TrailKit.UI.createSubtleMessage({ target = "all", header = "Goodbye!", message = playerObject.name .. " left the server.", options = { duration = 4 }})
    TrailKit.UI.removeSubtleMessage({ id = "player_hud", target = playerObject.id }) -- Cleanup HUD
end)

-- Runs when player state changes (e.g. "default" -> "building")
TrailKit.Event.on("onStateChange", function(playerObject, newState, oldState)
    TrailKit.Log.info(playerObject:getName() .. " changed state from '" .. oldState .. "' to '" .. newState .. "'")
    UpdateHUD(playerObject)
end)

-- Runs when entering a vehicle seat
TrailKit.Event.on("onEnterSeat", function(playerObject)
    playerObject:toast("You have entered a seat.", 3.0)
end)

-- Runs when leaving a seat
TrailKit.Event.on("onLeaveSeat", function(playerObject)
    playerObject:toast("You have left a seat.", 3.0)
end)

-- Runs when teleporting
TrailKit.Event.on("onTeleport", function(playerObject, from, to)
    playerObject:toast("You have teleported.", 3.0)
end)

-- Runs on respawn
TrailKit.Event.on("onRespawn", function(playerObject, type)
    playerObject:toast("You have respawned ("..type..").", 3.0)
end)

-- Runs when opening builder
TrailKit.Event.on("onEnterBuildMode", function(playerObject)
    playerObject:toast("You entered build mode.", 3.0)
end)

-- Runs when closing builder
TrailKit.Event.on("onLeaveBuildMode", function(playerObject)
    playerObject:toast("You left build mode.", 3.0)
end)

-- Runs when switching teams
TrailKit.Event.on("onTeamChange", function(playerObject, oldTeam, newTeam)
    playerObject:toast("You moved from team " .. oldTeam .. " to " .. newTeam, 4.0)
end)
